package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.base.StateConstant;
import com.example.afrishop_v3.bo.KeyConstant;
import com.example.afrishop_v3.config.RedisCache;
import com.example.afrishop_v3.enums.OrderStatusEnum;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.inter_face.OrderCount;
import com.example.afrishop_v3.models.*;
import com.example.afrishop_v3.repository.*;
import com.example.afrishop_v3.security.services.AuthenticationUser;
import com.example.afrishop_v3.util.IdUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.ParseException;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static org.springframework.data.domain.Sort.Order.asc;
import static org.springframework.data.domain.Sort.Order.desc;

@RestController
@RequestMapping("/order")
public class OrderController extends Controller {
    private final TbCfToicouponRepository couponRepository;
    private final TbCfOrderRepository repository;
    private final TbCfCartRecordRRepository cartRepository;
    private final TbCfAddressRepository addressRepository;
    private final TbCfToicouponRepository toicouponRepository;
    private final TbCfStationItemRepository itemRepository;
    private final TbCfItemSkuRepository itemSkuRepository;
    private final TbCfItemCommentRepository commentRepository;
    private final TbCfExpressTemplateRepository templateRepository;
    private final TbCfExchangeRepository exchangeRepository;
    private final ActivityRepository activityRepository;
    private final AuthenticationUser user;
    private final RedisCache redisCache;
    private static Logger logger = LoggerFactory.getLogger(OrderController.class);

    public OrderController(TbCfToicouponRepository couponRepository, TbCfOrderRepository repository, TbCfCartRecordRRepository cartRepository, TbCfAddressRepository addressRepository, TbCfToicouponRepository toicouponRepository, TbCfStationItemRepository itemRepository, TbCfItemSkuRepository itemSkuRepository, TbCfItemCommentRepository commentRepository, TbCfExpressTemplateRepository templateRepository, @Qualifier("tbCfExchangeRepository") TbCfExchangeRepository exchangeRepository, ActivityRepository activityRepository, AuthenticationUser user, RedisCache redisCache) {
        this.couponRepository = couponRepository;
        this.repository = repository;
        this.cartRepository = cartRepository;
        this.addressRepository = addressRepository;
        this.toicouponRepository = toicouponRepository;
        this.itemRepository = itemRepository;
        this.itemSkuRepository = itemSkuRepository;
        this.commentRepository = commentRepository;
        this.templateRepository = templateRepository;
        this.exchangeRepository = exchangeRepository;
        this.activityRepository = activityRepository;
        this.user = user;
        this.redisCache = redisCache;
    }


    //Simulate cart object for payNow process
    private TbCfCartRecordR getCart(TbCfStationItem item, String itemSku, Integer itemNum, BigDecimal price, TbCfItemSkus itemSkuObj) {

        TbCfCartRecordR record = new TbCfCartRecordR();

        String image = itemSkuObj.getSkuImg();

        record.setItemNum(itemNum);
        record.setItemId(item.getItemId());
        record.setCartRecordId(IdUtil.createIdbyUUID());
        record.setItemSku(itemSku);
        record.setItemSkuId(itemSkuObj.getId());
        record.setItemCount(itemSkuObj.getSkuCount() == null ? 0 : itemSkuObj.getSkuCount());
        record.setTemplate(item.getExpress());
        record.setItemTitle(item.getItemName());
        record.setItemPrice(price);
        record.setItemImg(image == null ? item.catchSingleImage() : image);

        return record;
    }


    @GetMapping("currencyConversion")
    public Result convert(@RequestParam("price") BigDecimal price, @RequestParam("currency") String currency) {
        TbCfExchange exchangeEntity = exchangeRepository.findByExchangeCurrency(currency.toUpperCase());
        if (exchangeEntity == null) {
            logger.warn("Currency not found");
            logger.warn(currency);
            return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(), "Currency Not Found");
        }
        BigDecimal rate = exchangeEntity.getExchangeRate();
        BigDecimal resultPrice = price.multiply(rate);
        return new Result<>(resultPrice.setScale(2, RoundingMode.CEILING));
    }


    @GetMapping("/payNow")
    public Result<TbCfOrder> payNow(@RequestParam("itemId") String itemId,
                                    @RequestParam("itemNum") Integer itemNum,
                                    @RequestParam(value = "itemSku", defaultValue = "") String itemSku,
                                    @RequestParam(value = "itemSkuId", required = false) String itemSkuId,
                                    @RequestParam(value = "toitableId", required = false) String toitableId,
                                    @RequestParam(value = "open", required = false) boolean open) throws ParseException {
        TbCfOrder order = new TbCfOrder();


        Optional<TbCfExpressTemplate> templateOptional = templateRepository.findFirstByIsDefault(1);
        templateOptional.ifPresent(order::setDefaultTemplate);

        int code = ResultCodeEnum.SERVICE_ERROR.getCode();
        if (StringUtils.isBlank(itemId) || itemNum == null || itemNum <= 0 || itemSkuId == null || itemSkuId.trim().isEmpty())
            return new Result<>(code, "Missing information !");

        Optional<TbCfStationItem> byId = itemRepository.findById(itemId);

        Optional<TbCfItemSkus> skuOptional = itemSkuRepository.findById(itemSkuId);


        if (!byId.isPresent())
            return new Result<>(code, "Item not found !");


        if (!skuOptional.isPresent())
            return new Result<>(code, "Sku not found !");

        TbCfStationItem item = byId.get();

        List<TbCfCartRecordR> list = new ArrayList<>();
        TbCfItemSkus itemSkus = skuOptional.get();
        BigDecimal orderPrice = itemSkus.getSkuPrice().multiply(new BigDecimal(itemNum));
        //校验优惠券是否可用
        if (toitableId != null && !toitableId.isEmpty()) {
            boolean check = checkCoupon(toitableId, orderPrice);
            if (!check) {
                return new Result<>(code, "Largest value Coupon selected");
            }
            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);


            if (couponOptional.isPresent()) {
                TbCfToicoupon tbCfToicoupon = couponOptional.get();
                TbCfCoupon coupon = couponOptional.get().getCoupon();
                order.setCoupon(coupon);
            }
        }
        Map<String, Object> map = queryUserAvailableCoupons(orderPrice);
        list.add(getCart(item, itemSku, itemNum, itemSkus.getSkuPrice(), itemSkus));

        //Check that one of items in the cart run out of stock
        //Check whether the inventory is greater than the purchased quantity before placing an order
        boolean anyMatch = list.stream().anyMatch(TbCfCartRecordR::isOutOfStock);

        if (anyMatch) return new Result<>(code, "Items run out of stock");

        order.setCouponId(toitableId);

        //Check coupon availability
//        if (toitableId != null && !toitableId.isEmpty()) {
//            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);
//
//
//            if (couponOptional.isPresent()) {
//                TbCfCoupon coupon = couponOptional.get().getCoupon();
//                order.setCoupon(coupon);
//            }
//        }
        boolean exists = activityRepository.existsByStatus(1);
        order.setHasActivity(exists);
        order.setOpen(open);
        order.setCouponMap(map);
        order.getItemOrderListFromCartList(list, itemRepository, activityRepository);

        return new Result<>(order);
    }

    public boolean checkCoupon(String couponId, BigDecimal orderPrice) {
        logger.info("orderPrice:" + orderPrice);
        logger.info("user:" + user.userId());
        logger.info("coupon:" + couponId);
        Optional<TbCfToicoupon> byId = toicouponRepository.checkCouponAvailable(user.userId(), couponId);
        if (byId.isPresent()) {
            TbCfToicoupon coupon = byId.get();
            logger.info("amout:" + coupon.deductAmount());
            System.out.println(orderPrice.compareTo(coupon.deductAmount()) >= 0);
            return orderPrice.compareTo(coupon.deductAmount()) >= 0;
        }
        return false;
    }

    @PostMapping("/settle")
    public Result<TbCfOrder> settleAccount(@RequestBody String[] ids, @RequestParam(value = "toitableId", required = false) String toitableId,
                                           @RequestParam(value = "open", required = false) boolean open) throws ParseException {
        //String userId = user.userId();
        List<TbCfCartRecordR> allByUserId = cartRepository.findAllByCartRecordIdIn(ids);
        BigDecimal orderPrice = BigDecimal.ZERO;

        for (TbCfCartRecordR cart : allByUserId) {
            orderPrice = orderPrice.add(cart.getItemPrice().multiply(new BigDecimal(cart.getItemNum())));
        }

        Map<String, Object> couponMap = queryUserAvailableCoupons(orderPrice);
        int code = ResultCodeEnum.VALIDATE_ERROR.getCode();
        //校验优惠券是否可用
        TbCfOrder order = new TbCfOrder();
        if (toitableId != null && !toitableId.isEmpty()) {
            boolean check = checkCoupon(toitableId, orderPrice);
            if (!check) {
                return new Result<>(code, "Largest value Coupon selected");
            }
            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);


            if (couponOptional.isPresent()) {
                TbCfCoupon coupon = couponOptional.get().getCoupon();
                order.setCoupon(coupon);
            }
        }
        if (allByUserId.isEmpty()) {
            return new Result<>(code, "There are no items in the shopping cart");
        }


        //Check that one of items in the cart run out of stock
        ////Check whether the inventory is greater than the purchased quantity before placing an order
        boolean anyMatch = allByUserId.stream().anyMatch(TbCfCartRecordR::isOutOfStock);

        if (anyMatch) return new Result<>(code, "Items run out of stock");

        order.setCouponMap(couponMap);
        Optional<TbCfExpressTemplate> templateOptional = templateRepository.findFirstByIsDefault(1);
        templateOptional.ifPresent(order::setDefaultTemplate);

        order.setCouponId(toitableId);

        //Check if any coupon selected
//        if (toitableId != null && !toitableId.isEmpty()) {
//            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);
//
//
//            if (couponOptional.isPresent()) {
//                TbCfCoupon coupon = couponOptional.get().getCoupon();
//                order.setCoupon(coupon);
//            }
//        }

        order.setOpen(open);
        order.getItemOrderListFromCartList(allByUserId, itemRepository, activityRepository);


        return new Result<>(order);
    }

    //Get server current time
    @GetMapping(value = "currentTime")
    public Result<Date> getTime() {
        return new Result<>(new Date());
    }


    @PostMapping("/place")
    @Transactional // Transaction for rollback if any failure
    public Result<TbCfOrder> placeOrder(@RequestBody OrderRequest tbCfOrder,
                                        @RequestParam(value = "toitableId", required = false) String toitableId,
                                        @RequestParam(value = "itemId", required = false) String itemId,
                                        @RequestParam(value = "itemNum", required = false) Integer itemNum,
                                        @RequestParam(value = "itemSku", required = false) String itemSku,
                                        @RequestParam(value = "itemSkuId", required = false) String itemSkuId,
                                        @RequestParam(value = "orderSource", required = false) Integer orderSource,
                                        @RequestParam(value = "open", required = false) boolean open
    ) throws ParseException {
        TbCfUserInfo user = this.user.user();
        if (user == null) {
            return new Result(ResultCodeEnum.UN_LOGIN.getCode(), "need login");
        }
        String userId = user.getUserId();

        int v_code = ResultCodeEnum.VALIDATE_ERROR.getCode();

        if (tbCfOrder == null)
            return new Result<>(v_code, "Empty body");

        boolean noCartBody = tbCfOrder.getIds() == null || tbCfOrder.getIds().length <= 0;

        boolean noPayNow = itemId == null || itemId.isEmpty() || itemNum == null || itemNum <= 0 || itemSkuId == null || itemSkuId.trim().isEmpty();

        if (noCartBody && noPayNow)
            return new Result<>(v_code, "Empty body");

        List<TbCfCartRecordR> allByUserId = new ArrayList<>();

        boolean payNow = !noPayNow;

        BigDecimal orderPrice = BigDecimal.ZERO;
        if (payNow) {

            Optional<TbCfStationItem> byId = itemRepository.findById(itemId);

            Optional<TbCfItemSkus> skuOptional = itemSkuRepository.findById(itemSkuId);


            if (!byId.isPresent())
                return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Item not found !");


            if (!skuOptional.isPresent())
                return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Sku not found !");
            TbCfStationItem stationItem = byId.get();
            //减库存 Inventory reduction
//            boolean success = subCount(itemId, itemNum);
//            if (!success) {
//                return new Result<>(v_code, "Goods in short stock!");
//            }

            TbCfItemSkus skuObj = skuOptional.get();
            orderPrice = skuObj.getSkuPrice().multiply(new BigDecimal(itemNum));
            allByUserId.add(getCart(stationItem, itemSku, itemNum, skuObj.getSkuPrice(), skuObj));

        } else {
            //System.out.println(Arrays.toString(tbCfOrder.getIds()));
            allByUserId = cartRepository.findAllByCartRecordIdIn(tbCfOrder.getIds());
//            for (TbCfCartRecordR cart : allByUserId) {
//                String id = cart.getItemId();
//                Integer num = cart.getItemNum();
//                boolean success = subCount(id, num);
//                if (!success) {
//                    return new Result<>(v_code, "Goods in short stock!");
//                }
//            }
            for (TbCfCartRecordR cart : allByUserId) {
                orderPrice = orderPrice.add(cart.getItemPrice().multiply(new BigDecimal(cart.getItemNum())));
            }
        }
        TbCfOrder order = new TbCfOrder();
        //校验优惠券是否可用
        if (toitableId != null && !toitableId.isEmpty()) {
            boolean check = checkCoupon(toitableId, orderPrice);
            if (!check) {
                return new Result<>(v_code, "Largest value Coupon selected");
            }
            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);


            if (couponOptional.isPresent()) {
                TbCfToicoupon tbCfToicoupon = couponOptional.get();
                tbCfToicoupon.setEnableFlag(0);
                toicouponRepository.save(tbCfToicoupon);
                TbCfCoupon coupon = couponOptional.get().getCoupon();
                order.setCoupon(coupon);
            }
        }
        Map<String, Object> map = queryUserAvailableCoupons(orderPrice);


        //Check that one of items in the cart run out of stock
        //Check whether the inventory is greater than the purchased quantity before placing an order
        boolean anyMatch = allByUserId.stream().anyMatch(TbCfCartRecordR::isOutOfStock);

        if (anyMatch) return new Result<>(v_code, "Items run out of stock");

        String addressId = tbCfOrder.getDeliveryAddressId();

        if (addressId == null || addressId.isEmpty()) return new Result<>(v_code, "Address id is required");

        Optional<TbCfAddress> addressOptional = addressRepository.findById(addressId);

        if (!addressOptional.isPresent()) return new Result<>(v_code, "Address id is required");


        if (allByUserId.isEmpty()) {
            return new Result<>(v_code, "There are no items in the shopping cart");
        }


        order.setCouponMap(map);
        order.setUserName(user.display());

        TbCfAddress address = addressOptional.get();

        String activityId = tbCfOrder.getActivityId();
        order.setDeliveryAddressId(addressId);
        order.setActivityId(activityId);
        order.setDeliveryAddress(address.getAddressDetail());
        order.setDeliveryName(address.getDeliveryName());
        order.setDeliveryPhone(address.getPhone());

        //Get default template
        Optional<TbCfExpressTemplate> templateOptional = templateRepository.findFirstByIsDefault(1);
        templateOptional.ifPresent(order::setDefaultTemplate);
        order.setOrderId(IdUtil.createIdbyUUID());


        order.setCouponId(toitableId);

        TbCfToicoupon tbCfToicoupon = null;
//        if (toitableId != null && !toitableId.isEmpty()) {
//            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);
//
//
//            if (couponOptional.isPresent()) {
//                tbCfToicoupon = couponOptional.get();
//                TbCfCoupon coupon = tbCfToicoupon.getCoupon();
//                order.setCoupon(coupon);
//            } else logger.warn("Invalid Coupon " + toitableId);
//        }


        order.setUserId(userId);


        if (orderSource == null) {
            order.setOrderSource(1);
        } else {
            order.setOrderSource(orderSource);
        }

        order.setOpen(open);
        order.getItemOrderListFromCartList(allByUserId, itemRepository, activityRepository);

        TbCfOrder save = repository.save(order);


        //implementation of coupon use
//        if (tbCfToicoupon != null) {
//            tbCfToicoupon.setEnableFlag(0);
//            toicouponRepository.save(tbCfToicoupon);
//        }

        //order.getItemOrderList().forEach(itemOrderRepository::save);

        List<String> collect = allByUserId.stream().map(TbCfCartRecordR::getCartRecordId).collect(Collectors.toList());

        //Convert to array
        String[] strings = collect.toArray(new String[]{});

        if (payNow) {

            //Automatically remove the shelf when the stock is insufficient
            cartRepository.updateItemShelf_(itemId, itemNum);
            //Inventory reduction for pay now item
            cartRepository.updateItemQuantity_(itemId, itemNum);
            cartRepository.updateSkuQuantity_(itemSkuId, itemNum);
        } else {
            //Automatically remove the shelf when the stock is insufficient
            cartRepository.updateItemsShelf(strings);
            //Inventory reduction for cart items
            cartRepository.updateSkuQuantity(strings);
            cartRepository.updateItemQuantity(strings);
        }
        cartRepository.updateCartCheckFlag(strings);


        // implementation of coupon use

        //Send notification to eligible devices
        if (user.hasFcm()) {
            sendNotification(user.getFcm(), "Order alert !!", "Order of $" + order.getRealityPay() + " has been created , proceed with payment");
        }

        return new Result<>(save);
    }

    //减库存 Inventory reduction
    public boolean subCount(String itemId, Integer num) {
        boolean success = false;
        logger.info("减库存操作开始...");
        Optional<TbCfStationItem> itemOptional = itemRepository.findById(itemId);
        if (itemOptional.isPresent()) {
            TbCfStationItem item = itemOptional.get();
            if (item.getItemCount() <= 0 || num > item.getItemCount()) {
                success = false;
            }
            logger.info("商品[" + item.getItemId() + "]原库存：" + item.getItemCount() + "，购买数量：" + num + "，现库存：" + (item.getItemCount() - num));
            item.setItemCount(item.getItemCount() - num);
            success = true;
        }
        return success;
    }

    @GetMapping
    public Result getUserOrderList(@RequestParam(value = "num", defaultValue = "0") Integer pageNum,
                                   @RequestParam(value = "size", defaultValue = "20") Integer pageSize,
                                   @RequestParam(value = "orderStatus", required = false) Integer orderStatus,
                                   @RequestParam(value = "sort", defaultValue = "desc") String sort,
                                   @RequestParam(value = "name", required = false) String name
    ) {
        TbCfUserInfo user = this.user.user();
        if (user == null) {
            return new Result(ResultCodeEnum.UN_LOGIN.getCode(), "need login");
        }
        Page<OrderCount> list;
        PageRequest of = PageRequest.of(pageNum, pageSize, sort(sort));
        String userId = user.getUserId();
        if (orderStatus != null) {
            list = repository.findAllByUserIdWithStatus(userId, user, orderStatus, of);
        } else
            list = repository.findAllByUserId(userId, user, of);

        list.forEach(v -> v.getOrder().setCommentCount(v.getCommented()));
        Page<TbCfOrder> map = list.map(OrderCount::getOrder);
        return new Result<>(map);
    }


    @GetMapping("/cancelOrder")
    @Transactional // transaction for rollback if something failed
    public Result cancelOrder(@RequestParam("orderId") String orderId,
                              @RequestParam("reason") String reason) {
        Optional<TbCfOrder> byId = repository.findById(orderId);

        if (byId.isPresent()) {
            TbCfOrder order = byId.get();
            order.setOrderStatus(OrderStatusEnum.CLOSE.getValue());
            order.setRemarkInfo(reason);

            if (order.getCouponId() != null) {
                Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(order.getCouponId());
                if (couponOptional.isPresent()) {
                    TbCfToicoupon toicoupon = couponOptional.get();
                    toicoupon.setEnableFlag(1);
                    toicouponRepository.save(toicoupon);
                }
            }
            //Return order items to the inventory
            repository.returnItemQuantity(orderId);
            repository.returnSkuQuantity(orderId);

            repository.save(order);

            return new Result();
        }


        return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Order not found!");
    }

    @DeleteMapping("/{orderId}")
    public Result deleteOrder(@PathVariable("orderId") String orderId) {

        Optional<TbCfOrder> byId = repository.findById(orderId);

        if (byId.isPresent()) {
            TbCfOrder order = byId.get();

            order.setEnableFlag(StateConstant.INVALID);

            System.out.println(orderId);

            logger.warn("Order deleted");

            repository.save(order);

            return new Result();
        }

        return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(), "Order not found");
    }


    @PostMapping("/add/comment")
    public Result addComment(@RequestBody TbCfItemComment comment) throws Exception {

        logger.info("上传files：", comment);

        TbCfUserInfo user = this.user.user();
        if (user == null) {
            return new Result(ResultCodeEnum.UN_LOGIN.getCode(), "need login");
        }

        String userId = user.getUserId();

        String itemId = comment.getItemId();

//        boolean exists = commentRepository.existsByUserUserIdAndItemId(userId, itemId);
//
//        if (exists) {
//            //Product reviewed condition
//            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(), "Already reviewed this product!");
//        }

//        List<String> urls = new ArrayList<>();
//        if (comment.getFiles() != null) {
//            for (MultipartFile file : comment.getFiles()) {
//                byte[] bytes = PicUtils.compressPicForScale(file.getBytes(), 30);
////                ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
//                String urlName = UUID.randomUUID() + ".jpg";
//                String url = OssUtil.upload(bytes, urlName, "comment");
//                urls.add(url);
//            }
//            String strUrl = urls.stream().collect(Collectors.joining(";"));
//            comment.setUrls(strUrl);
//        }

        comment.setLikeNum(0L);
        comment.setItemId(itemId);
        comment.setUser(user);
        comment.setId(IdUtil.createIdbyUUID());
        comment.setDelFlag(1);
//        comment.setLikeNum(0L);
        comment.setCreateTime(new Date());
        comment.setUpdateTime(new Date());
        comment.setCommentType(1);

//        comment.setType(0);
//        if (!StringUtils.isBlank(comment.getUrls())) {
//            comment.setType(1);
//        }

        TbCfItemComment save = commentRepository.save(comment);
        return new Result<>(save, "Comment on success!");
    }

    private Sort sort(String order) {
        String col = "orderTime";//Sort by order tim
        return Sort.by("desc".equals(order) ? desc(col) : asc(col));
    }


    @GetMapping("/details/{orderId}")
    public Result getOrderDetails(@PathVariable("orderId") String orderId) {
        Optional<TbCfOrder> optionalTbCfOrder = repository.findById(orderId);
        Date current = new Date();

        if (!optionalTbCfOrder.isPresent()) {
            return new Result<>(ResultCodeEnum.VALIDATE_ERROR.getCode(), "Order not found");
        }

        TbCfOrder tbCfOrder = optionalTbCfOrder.get();

//        Date orderTime = tbCfOrder.getOrderTime();
//
//
//        Calendar instance = Calendar.getInstance();
//        instance.setTime(orderTime);
//        instance.add(Calendar.DATE, 1);
//
//        long time = instance.getTime().getTime();
//
//        long difference = time - current.getTime();

        if (OrderStatusEnum.PENDING_PAY.getValue().equals(tbCfOrder.getOrderStatus())) {

//            Duration duration = Duration.ofMillis(difference);
//
//            long hours = duration.toHours();
//            long min = duration.minusHours(hours).toMinutes();
//            long sec = duration.minusHours(hours).minusMinutes(min).getSeconds();

// Or if you're lucky enough to be using Java 9+
//String formatted = String.format("%dhrs %02dmins", duration.toHours(), duration.toMinutesPart());
//            String formatted = String.format("%d:%02d:%02d", hours, min, sec);
            //因为运营需求，订单要求永不关闭，并且给定一个随机的倒计时
            Map<String, Object> randomTime = getRandomTime();
            String expire = (String) randomTime.get("expire");
            long seconds = (long) randomTime.get("seconds");
            redisCache.setNx(KeyConstant.ORDER_DET + "_" + orderId, expire, seconds);
            String exTime = redisCache.expire(KeyConstant.ORDER_DET + "_" + orderId);
            tbCfOrder.setExpiration(exTime);
        }
        return new Result<>(tbCfOrder, "success");
    }

    public Map<String, Object> getRandomTime() {
        Map<String, Object> map = new HashMap<>(2);
        Random rndHour = new Random();
        int hour = rndHour.nextInt(23);
        Random rndMinute = new Random();
        int minute = rndMinute.nextInt(60);
        Random rndSecond = new Random();
        int second = rndSecond.nextInt(60);
        String format = String.format("%d:%02d:%02d", hour, minute, second);
        long seconds = hour * 60 * 60 + minute * 60 + second;
        map.put("expire", format);
        map.put("seconds", seconds);
        return map;
    }

    public Map<String, Object> queryUserAvailableCoupons(BigDecimal orderPrice) {
        Map<String, Object> couponMap = new HashMap<>();
        //可使用的优惠券
        LinkedList<TbCfToicoupon> availableCoupon = new LinkedList<>();
        //不可使用的优惠券
        LinkedList<TbCfToicoupon> unAvailableCoupon = new LinkedList<>();
        List<TbCfToicoupon> couponList = couponRepository.queryUserAvailableCoupon(user.userId());
        couponList.forEach(coupon -> {
            boolean available = orderPrice.compareTo(coupon.withAmount()) >= 0;
            if (available) {
                availableCoupon.add(coupon);
            } else {
                unAvailableCoupon.add(coupon);
            }
        });
        couponMap.put("availableCoupon", availableCoupon);
        couponMap.put("unAvailableCoupon", unAvailableCoupon);

        return couponMap;
    }

}
