package com.example.afrishop_v3.controllers;


import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.config.PaypalPaymentIntent;
import com.example.afrishop_v3.config.PaypalPaymentMethod;
import com.example.afrishop_v3.enums.DeliveryStatusEnum;
import com.example.afrishop_v3.enums.OrderStatusEnum;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.models.*;
import com.example.afrishop_v3.repository.*;
import com.example.afrishop_v3.util.DateUtil;
import com.example.afrishop_v3.util.IdUtil;
import com.example.afrishop_v3.util.PayPalUtil;
import com.paypal.api.payments.*;
import com.paypal.base.rest.APIContext;
import com.paypal.base.rest.OAuthTokenCredential;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @Auther: wudepeng
 * @Date: 2020/11/27
 * @Description:paypal支付
 */
@RestController
@RequestMapping(value = "/paypal")
@Transactional
public class PaypalContoller extends Controller {

    private static Logger logger = LoggerFactory.getLogger(PaypalContoller.class);

    @Value("${paypal.success_url}")
    private String PAYPAL_SUCCESS_URL;

    @Value("${paypal.cancel_url}")
    private String PAYPAL_CANCEL_URL;

    @Value("${paypal.success_page}")
    private String PAYPAL_SUCCESS_PAGE;

    @Value("${paypal.failed_page}")
    private String PAYPAL_FAILED_PAGE;

    @Value("${paypal.mode}")
    private String mode;

    private final APIContext oldApiContext;
    private final TbCfOrderRepository orderRepository;
    private final TbCfFinanceRepository financeRepository;
    private final NetworkRepository networkRepository;
    private final BonusRepository bonusRepository;
    private final PostRepository postRepository;
    private final UserRepository userRepository;
    private final TokenRepository tokenRepository;


    public PaypalContoller(APIContext oldApiContext, TbCfOrderRepository orderRepository, TbCfFinanceRepository financeRepository, NetworkRepository networkRepository, BonusRepository bonusRepository, PostRepository postRepository, UserRepository userRepository, TokenRepository tokenRepository) {
        this.oldApiContext = oldApiContext;
        this.orderRepository = orderRepository;
        this.financeRepository = financeRepository;
        this.networkRepository = networkRepository;
        this.bonusRepository = bonusRepository;
        this.postRepository = postRepository;
        this.userRepository = userRepository;
        this.tokenRepository = tokenRepository;
    }


    /**
     * 发起paypal支付
     *
     * @param orderId 订单号
     * @return Result
     */
    @PostMapping("/payment/{orderId}")
    public Result payment(@PathVariable("orderId") String orderId) throws IOException {

        Result result = new Result();
        //==========================支付信息校验==========================
        //获取paypal的最新token
        APIContext apiContext = getToken();

        logger.info("APIContext info token:" + apiContext.getAccessToken());

        //订单号
        if (StringUtils.isBlank(orderId)) {
            logger.error("paypal支付，订单Id不能为空");
            return payErrorInfo(result);
        }

        //订单信息
        Optional<TbCfOrder> byId = orderRepository.findById(orderId);
        if (byId.isPresent()) {
            TbCfOrder order = byId.get();

            //订单不存在
            if (order == null) {
                logger.error("订单不存在");
                return payErrorInfo(result);
            }

            //价格不存在或错误
            if (order.getRealityPay() == null || order.getRealityPay().compareTo(BigDecimal.ZERO) <= 0) {
                logger.error("paypal支付，订单[" + order.getOrderNo() + "]价格错误");
                return payErrorInfo(result);
            }

            //订单已支付
            if (OrderStatusEnum.PAID.getValue().equals(order.getPayStatus().toString())) {
                logger.error("paypal支付，订单[" + order.getOrderNo() + "]已支付");
                return payErrorInfo(result);
            }

            //==========================paypal支付业务开始==========================

            logger.info("paypal支付开始，时间：" + getTime());
            String orderInfo = order.getDeliveryName() + "'s payment information";
            logger.info(orderInfo);
            try {

                Amount amount = new Amount();
                amount.setCurrency("USD");
                amount.setTotal(String.format("%.2f", order.getRealityPay()));

                Transaction transaction = new Transaction();
                transaction.setDescription(orderInfo);
                transaction.setAmount(amount);

                List<Transaction> transactions = new ArrayList<>();
                transactions.add(transaction);

                Payer payer = new Payer();
                payer.setPaymentMethod(PaypalPaymentMethod.paypal.toString());
                Payment payment = new Payment();
                payment.setIntent(PaypalPaymentIntent.sale.toString());
                payment.setPayer(payer);
                payment.setTransactions(transactions);
                RedirectUrls redirectUrls = new RedirectUrls();
                redirectUrls.setCancelUrl(PAYPAL_CANCEL_URL);
                redirectUrls.setReturnUrl(PAYPAL_SUCCESS_URL + "/" + orderId);
                payment.setRedirectUrls(redirectUrls);
                Payment pay = payment.create(apiContext);

                logger.info("paypal支付完成，时间：" + getTime());
                for (Links links : pay.getLinks()) {
                    if (links.getRel().equals("approval_url")) {
                        return result.setData(links.getHref());
                    }
                }
            } catch (Exception e) {
                logger.info("paypal支付发生异常，时间：" + getTime() + e.getMessage());
                return payErrorInfo(result);
            }
        }

        return payErrorInfo(result);
    }


    public APIContext getToken() throws IOException {
        //获取paypal的最新token
        Optional<Token> tokenOptional = tokenRepository.findFirstToken();
        String token = PayPalUtil.getToken(tokenOptional.get().getToken(), mode);
//        String token = PayPalUtil.getToken(oldApiContext.getAccessToken(), mode);
        APIContext apiContext = new APIContext(token);
        Map<String, String> sdkConfig = new HashMap<>();
        sdkConfig.put("mode", mode);
        apiContext.setConfigurationMap(sdkConfig);
        return apiContext;
    }

    /**
     * 用户取消支付
     */
    @GetMapping("/cancel")
    public void paymentCancle(HttpServletRequest request, HttpServletResponse response) throws IOException {
        response.sendRedirect(PAYPAL_FAILED_PAGE);
    }


    /**
     * 支付成功的回调方法
     * <p>
     * https://app.afrieshop.com/afrishop/paypal/success?
     * paymentId=PAYID-L6XFFIA6TM43947C4571820W&
     * token=EC-82206922L7926962C&
     * PayerID=WBS8FAZYEQQXS
     */
    @GetMapping("/success/{orderId}")
    public void paymentSuccess(@PathVariable("orderId") String orderId, HttpServletRequest request, HttpServletResponse response) throws Exception {
        logger.info("paypal支付校验开始，时间：" + getTime());
        //获取paypal的最新token
        APIContext apiContext = getToken();

        boolean verify = false;
        String payerId = request.getParameter("PayerID");
        //订单id
        String paymentId = request.getParameter("paymentId");
        String token = request.getParameter("token");
        System.out.println("payerId" + payerId);
        System.out.println("paymentId" + paymentId);
        System.out.println("token" + token);
        Payment payment = new Payment();
        payment.setId(paymentId);
        PaymentExecution paymentExecute = new PaymentExecution();
        paymentExecute.setPayerId(payerId);
        Payment paypal = payment.execute(apiContext, paymentExecute);

        logger.info("支付信息：" + paypal);
        logger.info("orderId:" + orderId);

        Optional<TbCfOrder> byId = orderRepository.findById(orderId);
        String payUrl = PAYPAL_SUCCESS_URL + "?paymentId=" + payerId + "&token=" + token + "&PayerID=" + payerId;
        if (byId.isPresent()) {
            TbCfOrder order = byId.get();
            //paypal支付成功
            if (paypal.getState().equals("approved")) {
                logger.info("paypal支付，订单[" + order.getOrderNo() + "]支付成功");

                //1)、修改订单状态
                TbCfOrder tbCfOrder = changeOrderState(payerId, order);

                //2)、生成支付流水
                createFinance(paymentId, payUrl, order);

                //数据库校验支付状态##PayStatus 20
                if (OrderStatusEnum.PAID.getValue().equals(tbCfOrder.getPayStatus()))
                    verify = true;
                //生成佣金
                Optional<TbCfOrder> optional = orderRepository.findById(orderId);
                if (optional.isPresent()) {
                    TbCfOrder order1 = optional.get();
                    Bonus bonus = new Bonus();
                    TbCfUserInfo userInfo = new TbCfUserInfo();
                    userInfo.setUserId(order1.getUserId());
//                        bonus.setUserId(tbCfOrder.getUserId());
                    bonus.setOrderId(orderId);
                    bonus.setAmount(order1.getItemsPrice());

                    System.out.println("佣金-----》》》订单号：" + orderId + "=user=" + order1.getUserId() + "=price=" + order1.getItemsPrice());
                    saveNetworkMarketing(bonus, order1.getUserId());
                }

            }
        }
        if (verify) {
            response.sendRedirect(PAYPAL_SUCCESS_PAGE);
        } else {
            response.sendRedirect(PAYPAL_FAILED_PAGE);
        }

    }

    /**
     * 支付失败提示
     *
     * @param result
     */
    public Result payErrorInfo(Result result) {
        result.setCode(ResultCodeEnum.ORDER_PAY_ERROR.getCode());
        result.setMessage(ResultCodeEnum.ORDER_PAY_ERROR.getDesc());
        return result;
    }


    /**
     * 修改订单状态
     */
    private TbCfOrder changeOrderState(String transToken, TbCfOrder order) {
        //更改订单状态
        order.setUpdateTime(DateUtil.getNow());
        order.setDealTime(DateUtil.getNow());
        order.setPayId(transToken);
        order.setOrderStatus(OrderStatusEnum.PAID.getValue());
        order.setPayStatus(OrderStatusEnum.PAID.getValue());
        order.setDeliveryFlag(DeliveryStatusEnum.PROCESSING.getValue());
        return orderRepository.save(order);
    }

    private TbCfFinance createFinance(String paymentId, String url, TbCfOrder tbCfOrderVo) {
        TbCfFinance tbCfFinance = new TbCfFinance();
        tbCfFinance.setOrderId(tbCfOrderVo.getOrderId());
        tbCfFinance.setFinaceId(IdUtil.createIdbyUUID());
        tbCfFinance.setPayAccount(tbCfOrderVo.getRealityPay());
        tbCfFinance.setPayId(paymentId);
        tbCfFinance.setPayTime(new Date());
        tbCfFinance.setReceiptUrl(url);
        tbCfFinance.setPayWayCode("paypal");
        tbCfFinance.setUserId(tbCfOrderVo.getUserId());
        financeRepository.save(tbCfFinance);
        return tbCfFinance;
    }


    public String getTime() {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        String now = format.format(new Date());
        return now;
    }

    public void saveNetworkMarketing(Bonus bonus, String userId) {

        String orderId = bonus.getOrderId();

        if (orderId == null)
            logger.info("佣金：orderId为空");

        Optional<TbCfOrder> orderOptional = orderRepository.findById(orderId);

        if (!orderOptional.isPresent())
            logger.info("佣金：订单不存在");

        TbCfOrder order = orderOptional.get();

        if (!OrderStatusEnum.PAID.getValue().equals(order.getPayStatus())) {
            logger.info("佣金：订单未支付");
        }

        boolean condition = orderId != null && orderOptional.isPresent() && OrderStatusEnum.PAID.getValue().equals(order.getPayStatus()) && !StringUtils.isBlank(userId);

//        if (bonusRepository.existsByOrderId(orderId)) {
//            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(), "Transaction already done !!!");
//        }


        BigDecimal amount = bonus.getAmount();

        if (condition) {


            Optional<TbCfUserInfo> optionalUser = userRepository.findById(userId);

            if (optionalUser.isPresent()) {

                TbCfUserInfo user = optionalUser.get();

                Post post = bonus.getPost();

                Optional<Post> postOptional = post == null ? Optional.empty() : postRepository.findById(post.getId());

                String productSharer = bonus.getProductSharer();

                Optional<TbCfUserInfo> sharer = Optional.empty();

                if (productSharer != null) {
                    sharer = userRepository.findByCode(productSharer);
                }
                synchronized (this) {
                    boolean exists = bonusRepository.existsByOrderId(orderId);
                    if (!exists && user.invited()) {
                        BigDecimal v = amount.multiply(BigDecimal.valueOf(10)).divide(BigDecimal.valueOf(100), RoundingMode.CEILING);
                        bonus.setAmount(v);
                        bonus.setUserInfo(user);
                        bonus.setPercentage(10);
                        bonus = bonusRepository.save(bonus);
                        if (user.hasFcm()) {
                            sendNotification(user.getFcm(), "Bonus alert !!", user.display() + ", You received bonus of $" + formatter.format(v) + " in your account");
                        }
                    }

                    TbCfUserInfo bonusInc = runBonusInc(user, amount, 5, false, orderId);
                    runBonusInc(sharer.orElseGet(() -> postOptional.isPresent() ? postOptional.get().getRealUser() : bonusInc), amount, 5, postOptional.isPresent() || sharer.isPresent(), orderId);
                    //runBonusInc(bonusInc, amount, 0);
                }
            }

        }


    }

    private TbCfUserInfo runBonusInc(TbCfUserInfo user, BigDecimal amount, int percent, boolean direct, String orderId) {
        if (user == null) return null;
        Optional<Network> userCode = networkRepository.findFirstByNetworkInfoCode(user.getCode());
        if (userCode.isPresent() || direct) {
            TbCfUserInfo userInfo = direct ? user : userCode.get().getUserInfo();
            Bonus bonus = new Bonus();
            bonus.setUserInfo(userInfo);
            BigDecimal v = amount.multiply(BigDecimal.valueOf(percent));
            v = v.divide(BigDecimal.valueOf(100), RoundingMode.CEILING);
            bonus.setAmount(v);
            bonus.setPercentage(percent);
            bonus.setOrderId(orderId);
            if (userInfo != null && userInfo.invited() && !"000000".equals(userInfo.getCode())) {
                bonusRepository.save(bonus);
//                bonus = repository.save(bonus);
                if (userInfo.hasFcm()) {
                    sendNotification(userInfo.getFcm(), "Bonus alert !!", userInfo.display() + ", You received bonus of $" + formatter.format(v) + " in your account");
                }
            }
            return userInfo;
        }

        return null;
    }
}
