package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.bo.KeyConstant;
import com.example.afrishop_v3.config.RedisCache;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.models.*;
import com.example.afrishop_v3.repository.*;
import com.example.afrishop_v3.security.services.AuthenticationUser;
import com.google.common.util.concurrent.AtomicDouble;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

import static org.springframework.data.domain.Sort.Order.*;

@RestController
@RequestMapping("/itemStation")
public class ItemController {
    private final TbCfStationItemRepository repository;
    private final TbCfGoodstwotypeRepository goodstwotypeRepository;
    private final TbCfGoodstypeRepository goodstypeRepository;
    private final TbCfDescripitonRepository descriptionRepository;
    private final TbCfCategoryRepository categoryRepository;
    private final TbCfItemSkuRepository skuRepository;
    private final TbCfItemDescRepository descRepository;
    private final TbCfItemCollectionRepository collectionRepository;
    private final TbCfItemParamRepository itemParamRepository;
    private final TbCfRecommendRepository recommendRepository;
    private final TbCfItemCommentRepository commentRepository;
    private final VirtualUserRepository virtualUserRepository;
    private final AuthenticationUser user;
    private final RedisCache redisCache;
    private final UserRepository userRepository;
    private static Logger logger = LoggerFactory.getLogger(ItemController.class);

    public ItemController(TbCfStationItemRepository repository, TbCfGoodstwotypeRepository goodstwotypeRepository, TbCfGoodstypeRepository goodstypeRepository, TbCfDescripitonRepository descriptionRepository, TbCfCategoryRepository categoryRepository, TbCfItemSkuRepository skuRepository, TbCfItemDescRepository descRepository, TbCfItemCollectionRepository collectionRepository, TbCfItemParamRepository itemParamRepository, TbCfRecommendRepository recommendRepository, TbCfItemCommentRepository commentRepository, VirtualUserRepository virtualUserRepository, AuthenticationUser user, RedisCache redisCache, UserRepository userRepository) {
        this.repository = repository;
        this.goodstwotypeRepository = goodstwotypeRepository;
        this.goodstypeRepository = goodstypeRepository;
        this.descriptionRepository = descriptionRepository;
        this.categoryRepository = categoryRepository;
        this.skuRepository = skuRepository;
        this.descRepository = descRepository;
        this.collectionRepository = collectionRepository;
        this.itemParamRepository = itemParamRepository;
        this.recommendRepository = recommendRepository;
        this.commentRepository = commentRepository;
        this.virtualUserRepository = virtualUserRepository;
        this.user = user;
        this.redisCache = redisCache;
        this.userRepository = userRepository;
    }

    @GetMapping("/queryAll")
    public Result queryAll(@RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                           @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                           @RequestParam(value = "order", required = false) String order) {

        return new Result<>(repository.findAllItems(PageRequest.of(pageNum, pageSize)));
    }


    @GetMapping("/queryItemsByTypeTwo")
    public Result queryItemsByTypeTwo(@RequestParam("typeTwoId") String typeTwoId,
                                      @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                      @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                                      @RequestParam(value = "order", required = false) String order) {

        return new Result<>(repository.findAllByItemCategorytwo(typeTwoId, PageRequest.of(pageNum, pageSize, getSort(order))));
    }

    @GetMapping("/queryCategoryTwoByCid/{categoryId}")
    public Result queryCategoryTwoByCid(@PathVariable("categoryId") String categoryId) {

        Map<String, Object> map = new HashMap<>();
        Optional<TbCfGoodstype> byId = goodstypeRepository.findById(categoryId);
        List<TbCfGoodstwotype> allByGoodstypeId = goodstwotypeRepository.findAllByGoodstypeIdOrderBySort(categoryId);
        map.put("category", byId.orElse(null));
        map.put("categoryTwoList", allByGoodstypeId);
        return new Result<>(map);
    }


    @GetMapping("/queryDescriptionByCTwoId/{categoryTwoId}")
    public Result queryDescriptionByCTwoId(@PathVariable("categoryTwoId") String categoryTwoId) {
        Map<String, Object> map = new HashMap<>();
        Optional<TbCfGoodstwotype> byId = goodstwotypeRepository.findById(categoryTwoId);
        List<TbCfDescripiton> allByGoodstwotypeId = descriptionRepository.findAllByGoodstwotypeIdOrderBySort(categoryTwoId);
        map.put("category", byId.orElse(null));
        map.put("descriptionList", allByGoodstwotypeId);
        return new Result<>(map);
    }

   /* @GetMapping("/searchItems")
    public Result searchItems(@RequestParam(value = "name") String name,
                              @RequestParam(value = "order", required = false) String order,
                              @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                              @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize) {


        try {
            List<SearchModel> list = repository.searchAllItems(name, pageNum * pageSize, pageSize);
            list = list.stream().filter(i -> "1".equals(i.getEnableFlag().toString())).collect(Collectors.toList());
            Long count = repository.countAllBySearchItems(name);
            long totalPage;
            if (count % pageSize == 0) {
                totalPage = count / pageSize;
            } else {
                totalPage = count / pageSize + 1;
            }
            Map<String, Object> map = new LinkedHashMap<>();
            map.put("content", list);
            map.put("totalPages", totalPage);
            return new Result<>(map);
        } catch (Exception e) {
            System.out.println(e.getMessage());
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(), e.getMessage());
        }
    }*/

    @GetMapping("/searchItems")
    public Result searchItems(@RequestParam(value = "name") String name,
                              @RequestParam(value = "order", required = false) String order,
                              @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                              @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize) {

        try {
            Page<TbCfStationItem> page = repository.findByItemNameLike(name.trim(), PageRequest.of(pageNum, pageSize, getSort(order)));
            return new Result<>(page);
        } catch (Exception e) {
            System.out.println(e.getMessage());
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(), e.getMessage());
        }
    }

    @GetMapping("/queryHotRecommended")
    public Result queryHotRecommended() {
        List<TbCfRecommend> all = (List<TbCfRecommend>) recommendRepository.findAll();
        return new Result<>(all.stream().map(TbCfRecommend::getRecommend).collect(Collectors.toList()));
    }


    @GetMapping("/getRecommendItems")
    public Result getRecommendItems(@RequestParam("itemId") String itemId,
                                    @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                    @RequestParam(value = "pageSize", defaultValue = "12") Integer pageSize) {

        Optional<TbCfStationItem> byId = repository.findById(itemId);
        if (byId.isPresent()) {

            TbCfStationItem stationItem = byId.get();

            Page<TbCfStationItem> recommendItems = repository.getRecommendItems(itemId, stationItem.getItemDescritionId(), stationItem.getItemCategorytwo(), stationItem.getItemCategory(), PageRequest.of(pageNum, pageSize));
            return new Result<>(recommendItems);
        } else logger.warn("Recommended item not found");
        return new Result<>(new ArrayList<>(), ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(), "Item id is invalid or not found");
    }

    @GetMapping("/queryItemByDescription/{descriptionId}")
    public Result queryItemByDescription(@PathVariable("descriptionId") String descriptionId,
                                         @RequestParam(value = "order", required = false) String order) {

        return new Result<>(repository.findAllByItemDescritionId(descriptionId, getSort(order)));
    }

    @GetMapping("/queryItemsByTypeThree")
    public Result queryItemsByTypeThree(@RequestParam("typeThreeId") String typeThreeId,
                                        @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                        @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                                        @RequestParam(value = "order", required = false) String order) {
        return new Result<>(repository.findAllByItemDescritionId(typeThreeId, PageRequest.of(pageNum, pageSize, getSort(order))));
    }

    private Sort getSort(String order) {
        return "priceUp".equals(order) ? sort("asc") : "priceDown".equals(order) ? sort("desc") : sort(order, "createTime");
    }

    private Sort sort(String order) {
        String col = "discountPrice";
        return sort(order, col);
    }

    private Sort sort(String order, String col) {
        return Sort.by(order(order, col));
    }

    private Sort.Order order(String order, String col) {
        return "desc".equals(order) ? desc(col) : asc(col);
    }

    private Sort sort2(String order) {
        String[] cols = new String[]{"itemTop", "discountPrice", "createTime"};
        List<Sort.Order> list = new ArrayList<>();

        for (String col : cols) {
            list.add(order(order, col));
        }

        return Sort.by(list);
    }

    @GetMapping("/queryItemsByLabel")
    public Result queryItemsByLabel(@RequestParam("label") String label,
                                    @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                    @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                                    @RequestParam(value = "order", required = false) String order) {
        return new Result<>(repository.findAllByItemLabelContaining(label, PageRequest.of(pageNum, pageSize, getSort(order))));
    }


    @GetMapping("/queryItemSku")
    public Result queryItemSku(@RequestParam(value = "itemId") String itemId,
                               @RequestParam(value = "userId", required = false) String userId) {

        Optional<TbCfStationItem> itemOptional = repository.findById(itemId);
        //Check if item is available
        if (!itemOptional.isPresent())
            return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(), "Item Id is not found");


        List<TbCfCategory> categoryList = categoryRepository.findAllByItemIdOrderByOrderNumAsc(itemId);
        List<TbCfItemSkus> skusList = skuRepository.findAllByItemIdOrderByOrderNumAsc(itemId);

        HashMap<String, Object> map = new HashMap<>();

        TbCfStationItem item = itemOptional.get();
        //商品参数
        List<TbCfItemParam> itemParamList = itemParamRepository.findAllByItemId(itemId);

        //商品详情
        List list = new ArrayList();
        Optional<TbCfItemDesc> byId = descRepository.findById(itemId);
        map.put("score", item.getTotalScore());
        map.put("itemScore", item.getItemScore());
        map.put("isCollection", userId != null && !userId.isEmpty() && collectionRepository.existsByUserIdAndItemItemId(userId, itemId));
        map.put("optionList", categoryList);
        map.put("itemDetail", skusList);
        map.put("itemInfo", item);
        map.put("itemParam", list);
        map.put("itemDesc", byId.orElse(new TbCfItemDesc()));

        return new Result<>(map);
    }

    @GetMapping("/queryItemComments")
    public Result queryItemComments(
            @RequestParam("itemId") String itemId,
            @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
            @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
            @RequestParam(value = "limit", required = false) Integer limit) {
        Result result = new Result();
        if (StringUtils.isBlank(itemId)) {
            result.setCode(ResultCodeEnum.VALIDATE_ERROR.getCode());
            result.setMessage("The itemId parameter cannot be empty");
            return result;
        }
        Map<String, Object> param = new HashMap();
        try {
            String userId = user.userId();

            String key = KeyConstant.ITEM_LIKE;
            Page<TbCfItemComment> allByItemId = commentRepository.findAllByItemId(itemId, PageRequest.of(pageNum, pageSize, Sort.by(Sort.Order.desc("createTime"))));
//            List<TbCfItemComment> list = new ArrayList<>();
            List<TbCfItemComment> comments = allByItemId.toList();

            AtomicDouble totalScore = new AtomicDouble(0d);
            if (comments.size() > 0 && comments != null) {
                if (limit != null) {
                    comments = comments.stream().limit(limit).collect(Collectors.toList());
                }

                comments.forEach(c -> {
//                    if (!StringUtils.isBlank(c.getUserId())) {
//                        if ("0".equals(c.getReal().toString())) {
//                            Optional<VirtualUser> byId = virtualUserRepository.findById(c.getUserId());
//                            if (byId.isPresent()) {
//                                VirtualUser user = byId.get();
//                                c.setAvatar(user.getAvatar());
//                                c.setUserName(user.getNick());
//                            }
//                        } else {
//                            Optional<TbCfUserInfo> byId = userRepository.findById(c.getUserId());
//                            if (byId.isPresent()) {
//                                TbCfUserInfo user = byId.get();
//                                c.setAvatar(user.getAvatar());
//                                c.setUserName(user.getNick());
//                            }
//                        }
//
//                    }

                    String nick = c.getUserName();
//                    String newNick = nick.substring(0, 1) + "***" + nick.substring(nick.length() - 1);
                    String newNick = String.format("%s***%s", nick.charAt(0), nick.charAt(nick.length() - 1));
                    c.setUserName(newNick);
                    if (!StringUtils.isBlank(userId)) {
                        String liked = (String) redisCache.get(key + userId + "_" + c.getId());
                        c.setLike(liked == null ? false : true);
                    }
                    Integer count = (Integer) redisCache.get(key + c.getId());
                    c.setLikeCount(count == null ? 0 : count);
                    totalScore.addAndGet(c.getScore());
//                    list.add(c);
                });
                double v = totalScore.get() / comments.size();
                param.put("totalScore", Double.parseDouble(String.format("%.1f", v)));
            }


            param.put("comments", comments);
            result.setData(param);
        } catch (Exception e) {
            result.setCode(ResultCodeEnum.SERVICE_ERROR.getCode());
            result.setMessage(ResultCodeEnum.SERVICE_ERROR.getDesc());
        }
        return result;
    }

    @GetMapping("/getItemLike/{commentId}")
    public Result getItemLike(@PathVariable("commentId") String commentId) {
        Result result = new Result();
        try {
            String userId = user.userId();
            if (StringUtils.isBlank(commentId)) {
                result.setCode(ResultCodeEnum.VALIDATE_ERROR.getCode());
                result.setMessage("The commentId parameter cannot be empty");
            }

            if (StringUtils.isBlank(userId)) {
                result.setCode(ResultCodeEnum.UN_LOGIN.getCode());
                result.setMessage(ResultCodeEnum.UN_LOGIN.getDesc());
            }

            String likeItemKey = KeyConstant.ITEM_LIKE + commentId;
            String userKey = KeyConstant.ITEM_LIKE + userId + "_" + commentId;

            Long likeNum = 0L;
            if (redisCache.setNx(userKey, likeItemKey)) {
                likeNum = redisCache.incr(likeItemKey);
            } else {
                redisCache.delete(userKey);
                likeNum = redisCache.decr(likeItemKey);
            }

            return result.setData(likeNum);
        } catch (Exception e) {
            result.setCode(ResultCodeEnum.SERVICE_ERROR.getCode());
            result.setMessage(ResultCodeEnum.SERVICE_ERROR.getDesc());
        }
        return result;
    }

}
