package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.enums.DeliveryStatusEnum;
import com.example.afrishop_v3.enums.OrderStatusEnum;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.models.*;
import com.example.afrishop_v3.repository.*;
import com.example.afrishop_v3.util.DateUtil;
import com.example.afrishop_v3.util.IdUtil;
import com.example.afrishop_v3.util.StripePay;
import com.stripe.exception.StripeException;
import com.stripe.model.Charge;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @Auther: wudepeng
 * @Date: 2021/01/04
 * @Description:Stripe支付
 */
@Transactional
@RestController
@RequestMapping("/stripe")
public class StripeController extends Controller {

    private static Logger logger = LoggerFactory.getLogger(StripeController.class);

    @Value("${stripe.secret_key}")
    private String SECRET_KEY;

    @Value("${stripe.public_key}")
    private String PUBLIC_KEY;

    @Value("${stripe.success_url}")
    private String SUCCESS_URL;

    @Value("${stripe.cancel_url}")
    private String CANCEL_URL;


    private final TbCfOrderRepository orderRepository;
    private final TbCfFinanceRepository financeRepository;
    private final NetworkRepository networkRepository;
    private final BonusRepository bonusRepository;
    private final PostRepository postRepository;
    private final UserRepository userRepository;

    public StripeController(TbCfOrderRepository orderRepository, TbCfFinanceRepository financeRepository, NetworkRepository networkRepository, BonusRepository bonusRepository, PostRepository postRepository, UserRepository userRepository) {
        this.orderRepository = orderRepository;
        this.financeRepository = financeRepository;
        this.networkRepository = networkRepository;
        this.bonusRepository = bonusRepository;
        this.postRepository = postRepository;
        this.userRepository = userRepository;
    }

    @PostMapping("/payment")
    public Result payment(@RequestParam Map<String, Object> params) throws StripeException {
        Result result = new Result(ResultCodeEnum.ORDER_PAY_ERROR.getCode(),ResultCodeEnum.ORDER_PAY_ERROR.getDesc());
        logger.info("stripe支付开始~~~");
        try {
            String orderId = (String) params.get("orderId");
            String token = (String) params.get("token");
            if (orderId == "" || orderId == null) {
                logger.info("订单号不能为空");
                return result;
            }
            if (token == "" || token == null) {
                logger.info("token不能为空");
                return result;
            }
//            boolean payed = false;
            Optional<TbCfOrder> byId = orderRepository.findById(orderId);
            if (byId.isPresent()) {
//                payed = true;
                TbCfOrder order = byId.get();
                BigDecimal orderPrice = order.getRealityPay();
                BigDecimal magnification = new BigDecimal("100");
                BigDecimal price = orderPrice.multiply(magnification);
                //支付业务
                Charge charge = StripePay.createCharge(price.longValue(), SECRET_KEY, token);
                logger.info("支付信息 %s:"+charge);
                if ("succeeded".equals(charge.getStatus())) {
                    logger.info("stripe支付,订单[" + orderId + "]支付成功~");
                    //支付成功
                    //1）、修改订单状态
                    changeOrderState(token, order);
                    //2）、生成支付流水
                    createFinance(charge.getId(), charge.getReceiptUrl(), order);
                    //3）、生成佣金
                    Bonus bonus = new Bonus();
                    bonus.setOrderId(orderId);
                    bonus.setAmount(order.getItemsPrice());

                    logger.info("佣金明细----->>>订单号=" + orderId + "user=" + order.getUserId() + "price=" + order.getItemsPrice());
                    saveNetworkMarketing(bonus, order.getUserId());
                    result.setCode(ResultCodeEnum.SUCCESS.getCode());
                    result.setMessage(ResultCodeEnum.SUCCESS.getDesc());
                }
            }

//            if (!payed) {
//                result.setCode(ResultCodeEnum.ORDER_PAY_ERROR.getCode());
//                result.setMessage(ResultCodeEnum.ORDER_PAY_ERROR.getDesc());
//                logger.info("stripe支付,订单[" + orderId + "]支付失败~");
//            }

        } catch (Exception e) {
            logger.error("stripe支付，发生异常：" + e.toString());


        }
        return result;
    }


    @GetMapping("/getPublicKey")
    public Result getPublicKey() {

        return new Result().setData(PUBLIC_KEY);
    }

    /**
     * 修改订单状态
     */
    private TbCfOrder changeOrderState(String transToken, TbCfOrder order) {
        //更改订单状态
        order.setUpdateTime(DateUtil.getNow());
        order.setDealTime(DateUtil.getNow());
        order.setPayId(transToken);
        order.setOrderStatus(OrderStatusEnum.PAID.getValue());
        order.setPayStatus(OrderStatusEnum.PAID.getValue());
        order.setDeliveryFlag(DeliveryStatusEnum.PROCESSING.getValue());
        return orderRepository.save(order);
    }

    private TbCfFinance createFinance(String paymentId, String url, TbCfOrder tbCfOrderVo) {
        TbCfFinance tbCfFinance = new TbCfFinance();
        tbCfFinance.setOrderId(tbCfOrderVo.getOrderId());
        tbCfFinance.setFinaceId(IdUtil.createIdbyUUID());
        tbCfFinance.setPayAccount(tbCfOrderVo.getRealityPay());
        tbCfFinance.setPayId(paymentId);
        tbCfFinance.setPayTime(new Date());
        tbCfFinance.setReceiptUrl(url);
        tbCfFinance.setPayWayCode("stripe");
        tbCfFinance.setUserId(tbCfOrderVo.getUserId());
        financeRepository.save(tbCfFinance);
        return tbCfFinance;
    }


    public String getTime() {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        String now = format.format(new Date());
        return now;
    }

    public void saveNetworkMarketing(Bonus bonus, String userId) {

        String orderId = bonus.getOrderId();

        if (orderId == null)
            logger.info("佣金：orderId为空");

        Optional<TbCfOrder> orderOptional = orderRepository.findById(orderId);

        if (!orderOptional.isPresent())
            logger.info("佣金：订单不存在");

        TbCfOrder order = orderOptional.get();

        if (!OrderStatusEnum.PAID.getValue().equals(order.getPayStatus())) {
            logger.info("佣金：订单未支付");
        }

        boolean condition = orderId != null && orderOptional.isPresent() && OrderStatusEnum.PAID.getValue().equals(order.getPayStatus()) && !StringUtils.isBlank(userId);

//        if (bonusRepository.existsByOrderId(orderId)) {
//            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(), "Transaction already done !!!");
//        }


        BigDecimal amount = bonus.getAmount();

        if (condition) {


            Optional<TbCfUserInfo> optionalUser = userRepository.findById(userId);

            if (optionalUser.isPresent()) {

                TbCfUserInfo user = optionalUser.get();

                Post post = bonus.getPost();

                Optional<Post> postOptional = post == null ? Optional.empty() : postRepository.findById(post.getId());

                String productSharer = bonus.getProductSharer();

                Optional<TbCfUserInfo> sharer = Optional.empty();

                if (productSharer != null) {
                    sharer = userRepository.findByCode(productSharer);
                }
                synchronized (this) {
                    boolean exists = bonusRepository.existsByOrderId(orderId);
                    if (!exists && user.invited()) {
                        BigDecimal v = amount.multiply(BigDecimal.valueOf(10)).divide(BigDecimal.valueOf(100), RoundingMode.CEILING);
                        bonus.setAmount(v);
                        bonus.setUserInfo(user);
                        bonus.setPercentage(10);
                        bonus = bonusRepository.save(bonus);
                        if (user.hasFcm()) {
                            sendNotification(user.getFcm(), "Bonus alert !!", user.display() + ", You received bonus of $" + formatter.format(v) + " in your account");
                        }
                    }

                    TbCfUserInfo bonusInc = runBonusInc(user, amount, 5, false, orderId);
                    if (sharer.isPresent() || postOptional.isPresent() || bonusInc != null)
                        runBonusInc(sharer.orElseGet(() -> postOptional.isPresent() ? postOptional.get().getRealUser() : bonusInc), amount, 5, postOptional.isPresent() || sharer.isPresent(), orderId);
                    //runBonusInc(bonusInc, amount, 0);
                }
            }

        }


    }

    private TbCfUserInfo runBonusInc(TbCfUserInfo user, BigDecimal amount, int percent, boolean direct, String orderId) {
        logger.info("user:" + user);
        Optional<Network> userCode = networkRepository.findFirstByNetworkInfoCode(user.getCode());
        if (userCode.isPresent() || direct) {
            TbCfUserInfo userInfo = direct ? user : userCode.get().getUserInfo();
            Bonus bonus = new Bonus();
            bonus.setUserInfo(userInfo);
            BigDecimal v = amount.multiply(BigDecimal.valueOf(percent));
            v = v.divide(BigDecimal.valueOf(100), RoundingMode.CEILING);
            bonus.setAmount(v);
            bonus.setPercentage(percent);
            bonus.setOrderId(orderId);
            if (userInfo != null && userInfo.invited() && !"000000".equals(userInfo.getCode())) {
                bonusRepository.save(bonus);
//                bonus = repository.save(bonus);
                if (userInfo.hasFcm()) {
                    sendNotification(userInfo.getFcm(), "Bonus alert !!", userInfo.display() + ", You received bonus of $" + formatter.format(v) + " in your account");
                }
            }
            return userInfo;
        }

        return null;
    }

}
