package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.models.ResetBody;
import com.example.afrishop_v3.models.TbCfUserInfo;
import com.example.afrishop_v3.repository.UserRepository;
import com.example.afrishop_v3.security.services.AuthenticationUser;
import com.example.afrishop_v3.util.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Consts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.web.bind.annotation.*;

import java.net.URLDecoder;
import java.util.Date;
import java.util.Optional;

@RestController
@RequestMapping("user")
public class UserController extends Controller {
    private final UserRepository repository;
    private final AuthenticationUser user;
    private final PasswordEncoder encoder;
    private final EmailHelper emailHelper;
    private static Logger logger = LoggerFactory.getLogger(UserController.class);

    public UserController(UserRepository repository, AuthenticationUser user, PasswordEncoder encoder, EmailHelper emailHelper) {
        this.repository = repository;
        this.user = user;
        this.encoder = encoder;
        this.emailHelper = emailHelper;
    }

    @PutMapping("bindPhoneOrEmail")
    public Result bindPhoneOrEmail(@RequestParam(value = "email",required = false) String email,@RequestParam(value = "code",required = false) String code,@RequestParam(value = "phone",required = false) String phone){
        TbCfUserInfo user = this.user.user();

        if( StringUtils.isNotBlank(email) ){

            boolean detect =  StringUtils.isNotBlank(code) && code.equals(user.getVerificationCode());
            if( isEmailValid(email) && detect ){
                boolean b = repository.existsByEmailAndUserIdNot(email, user.getUserId());
                if( b ){
                    return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),"Email is already taken");
                }
                if( user.getEmail() != null && user.getEmail().equals(user.getFirebaseUid())){
                    user.setFirebaseUid(email);
                }

                user.setEmail(email);
            }
            else return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(),detect ?"Email address is not valid" : "Code verification error");

        }


        if( StringUtils.isNotBlank(phone) ){

            if( StringUtils.isNotBlank(code) && code.equals(user.getVerificationCode()) ){
                boolean b = repository.existsByPhoneAndUserIdNot(phone, user.getUserId());
                if( b ){
                    return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),"Phone number is already taken");
                }
                user.setPhone(phone);
                user.setVerificationCode(null);
            }else return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),"Code verification error");

        }

        TbCfUserInfo save = repository.save(user);

        return new Result<>(save);
    }


    @GetMapping("/getEmailCode")
    public Result getEmailCode(@RequestParam("email") String email){

        try {
            EmailTemplateBo emailTemplateBo = new EmailTemplateBo();
            String name = URLDecoder.decode(email, Consts.UTF_8.name());
            Integer randomCode = RandomCodeHelper.producedRandomCode(6);
            emailTemplateBo.setNick(name);
            emailTemplateBo.setIdentifyCode(randomCode);
            Integer identifyCode = emailHelper.sendIdentifyEmail(email, EmailTemplateConstant.REGISTER, emailTemplateBo);
            TbCfUserInfo user = this.user.user();
            user.setVerificationCode(String.valueOf(identifyCode));
            repository.save(user);
            return new Result("Verification code has been sent");
        }catch (Exception e){
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),e.getMessage());
        }
    }

    @GetMapping("/identifyCode/{phone}")
    public Result getUserPhoneCode(@PathVariable("phone") String phone) {
        TbCfUserInfo user = this.user.user();

        Result result = new Result();
        try {
            //手机号拼接"+"
            phone = "+" + phone.trim();
            /**
             * 验证码时效5分钟，如果用户连续发送3次请求，则提示用户：操作频繁，请15分钟后重试
             */
            //先从缓存中获取用户的发送次数
            int sentCount = user.getSentCount();

            System.err.println(sentCount);
            //用户请求次数小于3次，则正常发送验证码，并且有效期为5分钟

            Date date = new Date();

            Date sentTime = user.getCodeSentTime();


            long diff = date.getTime() - (sentTime == null ? 0 : sentTime.getTime());

            long diffMinutes = diff / (60 * 1000) % 60;

            if( diffMinutes > 15 && sentCount >= 3 ){
                sentCount = -1;
                user.setSentCount(sentCount);
            }

            boolean canSend = sentCount < 3;


            if ( canSend ) {
                result.setMessage("Verification code has been sent");
                //发送验证码
                String code = SMSUtil.yzCode(phone);
                user.setCodeSentTime(date);
                user.setVerificationCode(code);
                user.setSentCount(user.getSentCount()+1);
                repository.save(user);
                logger.info("用户[" + phone + "]获取验证码成功，" + "验证码：" + code);
            } else {
                //用户频繁操作
                result.setCode(ResultCodeEnum.SERVICE_ERROR.getCode());
                result.setMessage("Frequent operation, please try again after 15 minutes");
                logger.info("用户[" + phone + "]获取验证码失败，操作频繁，15分钟后重试");
            }
        } catch (Exception e) {
            result.setCode(ResultCodeEnum.SERVICE_ERROR.getCode());
            result.setMessage(e.toString());
            logger.info("用户[" + phone + "]获取验证码发生异常--->>>" + e.toString());
        }
        return result;
    }

    @PutMapping("updatePassword")
    public Result updatePassword(@RequestBody ResetBody body){

        if( body == null || body.getNewPassword() == null || body.getOldPassword() == null || body.getOldPassword().isEmpty() || body.getNewPassword().isEmpty()){
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),"Empty body");
        }


        TbCfUserInfo user = this.user.user();


        boolean matches = encoder.matches(body.getOldPassword(), user.getPassword());

        if( !matches ){
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),"Old password does not match");
        }

        user.setPassword(encoder.encode(body.getNewPassword()));

        repository.save(user);

        return new Result<>(user);

    }

    @PutMapping
    public Result updateUser(@RequestBody TbCfUserInfo info) {
        TbCfUserInfo user = this.user.user();

        if (info == null) return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(), "Empty body");

        if(StringUtils.isNotBlank(info.getAvatar()) ){
            user.setAvatar(info.getAvatar());
        }

        if(StringUtils.isNotBlank(info.getBirthday()) ){
            user.setBirthday(info.getBirthday());
        }

        if(StringUtils.isNotBlank(info.getNick()) ){
            user.setNick(info.getNick());
        }

        if( info.getSex() != null ){
            user.setSex(info.getSex());
        }


        TbCfUserInfo save = repository.save(user);


        return new Result<>(save);

    }

    @GetMapping("/userById/{id}")
    public Result userById(@PathVariable("id") String id){
        Optional<TbCfUserInfo> byId = repository.findById(id);
        return byId.map(userInfo -> new Result<>(userInfo, "Success")).orElseGet(() -> new Result<>(ResultCodeEnum.VALIDATE_ERROR.getCode(), "Not found !"));

    }

    @GetMapping("/sendNotification/{id}")
    public Result sendFcm(@PathVariable("id") String id,@RequestParam("content") String content,@RequestParam("title") String title){
        Optional<TbCfUserInfo> byId = repository.findById(id);

        if( byId.isPresent() && byId.get().hasFcm() ){
            TbCfUserInfo userInfo = byId.get();
            sendNotification(userInfo.getFcm(),title,content);
            return new Result();
        }

        return new Result(ResultCodeEnum.SUCCESS.getCode(),"No firebase messaging token found");
    }


    @GetMapping(value = "/userByCode/{code}")
    public Result getUserByCode(@PathVariable("code") String code) {
        Optional<TbCfUserInfo> byCode = repository.findByCode(code);
        boolean present = byCode.isPresent();
        return new Result<>(present ? byCode.get() : null,present ? 1 : 0, present ? "Successfully" : "Invitation code not found");
    }
    @PostMapping(value = "/edit/slogan")
    public Result editSlogan(@RequestBody TbCfUserInfo tbCfUserInfo) {

        TbCfUserInfo user = this.user.user();

        if( user != null && tbCfUserInfo != null){
            user.setSlogan(tbCfUserInfo.getSlogan());
            repository.save(user);
        }

        return new Result<>(user);
    }
}
