package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.base.StateConstant;
import com.example.afrishop_v3.enums.OrderStatusEnum;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.inter_face.OrderCount;
import com.example.afrishop_v3.models.*;
import com.example.afrishop_v3.repository.*;
import com.example.afrishop_v3.security.services.AuthenticationUser;
import com.example.afrishop_v3.util.IdUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URISyntaxException;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import static org.springframework.data.domain.Sort.Order.asc;
import static org.springframework.data.domain.Sort.Order.desc;

@RestController
@RequestMapping("/order")
public class OrderController extends Controller {
    private final TbCfOrderRepository repository;
    private final TbCfCartRecordRRepository cartRepository;
    private final TbCfToicouponRepository toicouponRepository;
    private final TbCfStationItemRepository itemRepository;
    private final TbCfItemCommentRepository commentRepository;
    private final TbCfExpressTemplateRepository templateRepository;
    private final TbCfExchangeRepository exchangeRepository;
    private final AuthenticationUser user;

    public OrderController(TbCfOrderRepository repository, TbCfCartRecordRRepository cartRepository, TbCfToicouponRepository toicouponRepository, TbCfStationItemRepository itemRepository, TbCfItemCommentRepository commentRepository, TbCfExpressTemplateRepository templateRepository, @Qualifier("tbCfExchangeRepository") TbCfExchangeRepository exchangeRepository, AuthenticationUser user) {
        this.repository = repository;
        this.cartRepository = cartRepository;
        this.toicouponRepository = toicouponRepository;
        this.itemRepository = itemRepository;
        this.commentRepository = commentRepository;
        this.templateRepository = templateRepository;
        this.exchangeRepository = exchangeRepository;
        this.user = user;
    }

    private TbCfCartRecordR getCart(TbCfStationItem item, String itemSku, Integer itemNum,Double price) {

        TbCfCartRecordR record = new TbCfCartRecordR();

        record.setItemNum(itemNum);
        record.setItemId(item.getItemId());
        record.setCartRecordId(IdUtil.createIdbyUUID());
        record.setItemSku(itemSku);
        record.setTemplate(item.getExpress());
        record.setItemTitle(item.getItemName());
        record.setItemPrice(BigDecimal.valueOf(price));
        record.setItemImg(item.catchSingleImage());

        return record;
    }


    @GetMapping("currencyConversion")
    public Result convert(@RequestParam("price") BigDecimal price,@RequestParam("currency") String currency){
        TbCfExchange exchangeEntity = exchangeRepository.findByExchangeCurrency(currency.toUpperCase());
        if( exchangeEntity == null){
            return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(),"Currency Not Found");
        }
        BigDecimal rate = exchangeEntity.getExchangeRate();
        BigDecimal resultPrice = price.multiply(rate);
        return new Result<>(resultPrice.setScale(2, RoundingMode.CEILING));
    }


    @GetMapping("/payNow")
    public Result<TbCfOrder> payNow(@RequestParam("itemId") String itemId,
                                    @RequestParam("itemNum") Integer itemNum,
                                    @RequestParam(value = "itemSku",defaultValue = "") String itemSku,
                                    @RequestParam(value = "itemPrice") Double itemPrice,
                                    @RequestParam(value = "toitableId", required = false) String toitableId) {
        TbCfOrder order = new TbCfOrder();

        Optional<TbCfExpressTemplate> templateOptional = templateRepository.findFirstByIsDefault(1);
        templateOptional.ifPresent(order::setDefaultTemplate);

        if (StringUtils.isBlank(itemId) || itemNum == null || itemNum <= 0 || itemPrice == null || itemPrice <= 0)
            return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Missing information !");

        Optional<TbCfStationItem> byId = itemRepository.findById(itemId);

        if (!byId.isPresent())
            return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Missing information !");

        TbCfStationItem item = byId.get();


        List<TbCfCartRecordR> list = new ArrayList<>();
        list.add(getCart(item, itemSku, itemNum,itemPrice));
        order.setCouponId(toitableId);

        if (toitableId != null && !toitableId.isEmpty()) {
            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);


            if( couponOptional.isPresent() ){
                TbCfCoupon coupon = couponOptional.get().getCoupon();
                order.setCoupon(coupon);
            }
        }


        order.getItemOrderListFromCartList(list);

        return new Result<>(order);
    }

    @PostMapping("/settle")
    public Result<TbCfOrder> settleAccount(@RequestBody String[] ids, @RequestParam(value = "toitableId", required = false) String toitableId) {
        //String userId = user.userId();
        List<TbCfCartRecordR> allByUserId = cartRepository.findAllByCartRecordIdIn(ids);

        if (allByUserId.isEmpty()) {
            return new Result<>(ResultCodeEnum.VALIDATE_ERROR.getCode(), "There are no items in the shopping cart");
        }

        TbCfOrder order = new TbCfOrder();

        Optional<TbCfExpressTemplate> templateOptional = templateRepository.findFirstByIsDefault(1);
        templateOptional.ifPresent(order::setDefaultTemplate);

        order.setCouponId(toitableId);

        if (toitableId != null && !toitableId.isEmpty()) {
            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);


            if( couponOptional.isPresent() ){
                TbCfCoupon coupon = couponOptional.get().getCoupon();
                order.setCoupon(coupon);
            }
        }


        order.getItemOrderListFromCartList(allByUserId);


        return new Result<>(order);
    }

    @GetMapping(value = "currentTime")
    public Result<Date> getTime(){
        return new Result<>(new Date());
    }


    @PostMapping("/place")
    public Result<TbCfOrder> placeOrder(@RequestBody OrderRequest tbCfOrder,
                                        @RequestParam(value = "toitableId", required = false) String toitableId,
                                        @RequestParam(value = "itemId", required = false) String itemId,
                                        @RequestParam(value = "itemNum",required = false) Integer itemNum,
                                        @RequestParam(value = "itemSku",required = false) String itemSku,
                                        @RequestParam(value = "itemPrice",required = false) Double itemPrice,
                                        @RequestParam(value = "web", required = false) String web) throws IOException, URISyntaxException, ExecutionException, InterruptedException, TimeoutException {
        TbCfUserInfo user = this.user.user();
        String userId = user.getUserId();

        int v_code = ResultCodeEnum.VALIDATE_ERROR.getCode();

        if (tbCfOrder == null)
            return new Result<>(v_code, "Empty body");

        boolean noCartBody = tbCfOrder.getIds() == null || tbCfOrder.getIds().length <= 0;

        boolean noPayNow = itemId == null || itemId.isEmpty() || itemNum == null || itemNum <= 0 || itemPrice == null || itemPrice <= 0;

        if (noCartBody && noPayNow)
            return new Result<>(v_code, "Empty body");

        List<TbCfCartRecordR> allByUserId = new ArrayList<>();
        if (!noPayNow) {
            Optional<TbCfStationItem> byId = itemRepository.findById(itemId);

            if (!byId.isPresent())
                return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Missing information !");

            TbCfStationItem stationItem = byId.get();

            allByUserId.add(getCart(stationItem, itemSku,itemNum,itemPrice));

        } else {
            System.out.println(Arrays.toString(tbCfOrder.getIds()));
            allByUserId = cartRepository.findAllByCartRecordIdIn(tbCfOrder.getIds());
        }

        String addressId = tbCfOrder.getDeliveryAddressId();

        if (addressId == null || addressId.isEmpty()) return new Result<>(v_code, "Address id is required");


        if (allByUserId.isEmpty()) {
            return new Result<>(v_code, "There are no items in the shopping cart");
        }


        TbCfOrder order = new TbCfOrder();

        order.setUserName(user.display());


        Optional<TbCfExpressTemplate> templateOptional = templateRepository.findFirstByIsDefault(1);
        templateOptional.ifPresent(order::setDefaultTemplate);
        order.setOrderId(IdUtil.createIdbyUUID());


        order.setCouponId(toitableId);

        TbCfToicoupon tbCfToicoupon = null;
        if (toitableId != null && !toitableId.isEmpty()) {
            Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(toitableId);


            if( couponOptional.isPresent() ){
                tbCfToicoupon = couponOptional.get();
                TbCfCoupon coupon = tbCfToicoupon.getCoupon();
                order.setCoupon(coupon);
            }
        }


        order.setUserId(userId);

        if ("web".equals(web)) {
            order.setOrderSource(3);
        } else {
            order.setOrderSource(1);
        }


        order.getItemOrderListFromCartList(allByUserId);

        TbCfOrder save = repository.save(order);

        if( tbCfToicoupon != null ){
            tbCfToicoupon.setEnableFlag(0);
            toicouponRepository.save(tbCfToicoupon);
        }

        //order.getItemOrderList().forEach(itemOrderRepository::save);

        List<String> collect = allByUserId.stream().map(TbCfCartRecordR::getCartRecordId).collect(Collectors.toList());

        String[] strings = collect.toArray(new String[]{});
        cartRepository.deleteAllByCartRecordIdIn(strings);


        // implementation of coupon use


        if( user.hasFcm() ){
            sendNotification(user.getFcm(),"Order alert !!","Order of $"+order.getRealityPay()+" has been created , proceed with payment");
        }

        return new Result<>(save);
    }

    @GetMapping
    public Result getUserOrderList(@RequestParam(value = "num", defaultValue = "0") Integer pageNum,
                                   @RequestParam(value = "size", defaultValue = "20") Integer pageSize,
                                   @RequestParam(value = "orderStatus", required = false) Integer orderStatus,
                                   @RequestParam(value = "sort", defaultValue = "desc") String sort,
                                   @RequestParam(value = "name", required = false) String name
    ) {
        TbCfUserInfo user = this.user.user();
        Page<OrderCount> list = repository.findAllByUserId(user.getUserId(), user, PageRequest.of(pageNum, pageSize, sort(sort)));
        list.forEach(v->v.getOrder().setCommentCount(v.getCommented()));
        Page<TbCfOrder> map = list.map(OrderCount::getOrder);
        return new Result<>(map);
    }


    @GetMapping("/cancelOrder")
    public Result cancelOrder(@RequestParam("orderId") String orderId,
                              @RequestParam("reason") String reason) {
        Optional<TbCfOrder> byId = repository.findById(orderId);

        if (byId.isPresent()) {
            TbCfOrder order = byId.get();
            order.setOrderStatus(OrderStatusEnum.CLOSE.getValue());
            order.setRemarkInfo(reason);

            if( order.getCouponId() != null ){
                Optional<TbCfToicoupon> couponOptional = toicouponRepository.findById(order.getCouponId());
                if ( couponOptional.isPresent() ){
                    TbCfToicoupon toicoupon = couponOptional.get();
                    toicoupon.setEnableFlag(1);
                    toicouponRepository.save(toicoupon);
                }
            }

            repository.save(order);

            return new Result();
        }


        return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode(), "Order not found!");
    }

    @DeleteMapping("/{orderId}")
    public Result deleteOrder(@PathVariable("orderId") String orderId) {

        Optional<TbCfOrder> byId = repository.findById(orderId);

        if (byId.isPresent()) {
            TbCfOrder order = byId.get();

            order.setEnableFlag(StateConstant.INVALID);

            System.out.println(orderId);

            repository.save(order);

            return new Result();
        }

        return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(), "Order not found");
    }


    @PostMapping("/add/comment")
    public Result addComment(@RequestBody TbCfItemComment comment) {

        TbCfUserInfo user = this.user.user();

        String userId = user.getUserId();

        String itemId = comment.getItemId();

        boolean exists = commentRepository.existsByUserUserIdAndItemId(userId, itemId);

        if( exists ){
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),"Already reviewed this product!");
        }

        comment.setItemId(itemId);
        comment.setUser(user);
        comment.setId(IdUtil.createIdbyUUID());
        comment.setDelFlag(0);
        comment.setLikeNum(0L);
        comment.setCreateTime(new Date());
        comment.setUpdateTime(new Date());
        comment.setType(0);
        if (!StringUtils.isBlank(comment.getUrls())) {
            comment.setType(1);
        }

        TbCfItemComment save = commentRepository.save(comment);
        return new Result<>(save,"Comment on success!");
    }

    private Sort sort(String order) {
        String col = "orderTime";
        return Sort.by("desc".equals(order) ? desc(col) : asc(col));
    }

}
