package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.models.*;
import com.example.afrishop_v3.repository.*;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

import static org.springframework.data.domain.Sort.Order.*;

@RestController
@RequestMapping("/itemStation")
public class ItemController {
    private final TbCfStationItemRepository repository;
    private final TbCfGoodstwotypeRepository goodstwotypeRepository;
    private final TbCfGoodstypeRepository goodstypeRepository;
    private final TbCfDescripitonRepository descriptionRepository;
    private final TbCfCategoryRepository categoryRepository;
    private final TbCfItemSkuRepository skuRepository;
    private final TbCfItemDescRepository descRepository;
    private final TbCfItemCollectionRepository collectionRepository;
    private final TbCfItemParamRepository itemParamRepository;
    private final TbCfRecommendRepository recommendRepository;

    public ItemController(TbCfStationItemRepository repository, TbCfGoodstwotypeRepository goodstwotypeRepository, TbCfGoodstypeRepository goodstypeRepository, TbCfDescripitonRepository descriptionRepository, TbCfCategoryRepository categoryRepository, TbCfItemSkuRepository skuRepository, TbCfItemDescRepository descRepository, TbCfItemCollectionRepository collectionRepository, TbCfItemParamRepository itemParamRepository, TbCfRecommendRepository recommendRepository) {
        this.repository = repository;
        this.goodstwotypeRepository = goodstwotypeRepository;
        this.goodstypeRepository = goodstypeRepository;
        this.descriptionRepository = descriptionRepository;
        this.categoryRepository = categoryRepository;
        this.skuRepository = skuRepository;
        this.descRepository = descRepository;
        this.collectionRepository = collectionRepository;
        this.itemParamRepository = itemParamRepository;
        this.recommendRepository = recommendRepository;
    }

    @GetMapping("/queryAll")
    public Result queryAll(@RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                           @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                           @RequestParam(value = "order", required = false) String order) {

        return new Result<>(repository.findAllItems(PageRequest.of(pageNum,pageSize)));
    }


    @GetMapping("/queryItemsByTypeTwo")
    public Result queryItemsByTypeTwo( @RequestParam("typeTwoId") String typeTwoId,
                                      @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                      @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                                      @RequestParam(value = "order", required = false) String order) {

        return new Result<>(repository.findAllByItemCategorytwo(typeTwoId, PageRequest.of(pageNum,pageSize,sort(order))));
    }

    @GetMapping("/queryCategoryTwoByCid/{categoryId}")
    public Result queryCategoryTwoByCid( @PathVariable("categoryId") String categoryId) {

        Map<String, Object> map = new HashMap<>();
        Optional<TbCfGoodstype> byId = goodstypeRepository.findById(categoryId);
        List<TbCfGoodstwotype> allByGoodstypeId = goodstwotypeRepository.findAllByGoodstypeId(categoryId);
        map.put("category", byId.orElse(null));
        map.put("categoryTwoList", allByGoodstypeId);
        return new Result<>(map);
    }


    @GetMapping("/queryDescriptionByCTwoId/{categoryTwoId}")
    public Result queryDescriptionByCTwoId(@PathVariable("categoryTwoId") String categoryTwoId) {
        Map<String, Object> map = new HashMap<>();
        Optional<TbCfGoodstwotype> byId = goodstwotypeRepository.findById(categoryTwoId);
        List<TbCfDescripiton> allByGoodstwotypeId = descriptionRepository.findAllByGoodstwotypeId(categoryTwoId);
        map.put("category", byId.orElse(null));
        map.put("descriptionList", allByGoodstwotypeId);
        return new Result<>(map);
    }

    @GetMapping("/searchItems")
    public Result searchItems(@RequestParam(value = "name") String name,
                              @RequestParam(value = "order", required = false) String order,
                              @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                              @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize) {


        try {
            List<SearchModel> list = repository.searchAllItems(name,pageNum*pageSize,pageSize);
            Map<String,Object> map = new LinkedHashMap<>();
            map.put("content",list);
            return new Result<>(map);
        }catch (Exception e){
            System.out.println(e.getMessage());
            return new Result(ResultCodeEnum.VALIDATE_ERROR.getCode(),e.getMessage());
        }
    }

    @GetMapping("/queryHotRecommended")
    public Result queryHotRecommended() {
        List<TbCfRecommend> all = (List<TbCfRecommend>) recommendRepository.findAll();
        return new Result<>(all.stream().map(TbCfRecommend::getRecommend).collect(Collectors.toList()));
    }


    @GetMapping("/getRecommendItems")
    public Result getRecommendItems(@RequestParam("itemId") String itemId,
                                    @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                    @RequestParam(value = "pageSize", defaultValue = "12") Integer pageSize) {

        Optional<TbCfStationItem> byId = repository.findById(itemId);
        if( byId.isPresent() ) {

            TbCfStationItem stationItem = byId.get();

            Page<TbCfStationItem> recommendItems = repository.getRecommendItems(itemId, stationItem.getItemDescritionId(), stationItem.getItemCategorytwo(), stationItem.getItemCategory(), PageRequest.of(pageNum, pageSize));
            return new Result<>(recommendItems);
        }
        return new Result<>(new ArrayList<>(),ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(),"Item id is invalid or not found");
    }

    @GetMapping("/queryItemByDescription/{descriptionId}")
    public Result queryItemByDescription(@PathVariable("descriptionId") String descriptionId) {

        return new Result<>(repository.findAllByItemDescritionId(descriptionId));
    }

    @GetMapping("/queryItemsByTypeThree")
    public Result queryItemsByTypeThree( @RequestParam("typeThreeId") String typeThreeId,
                                         @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                        @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                                        @RequestParam(value = "order", required = false) String order) {
        return new Result<>(repository.findAllByItemDescritionId(typeThreeId, PageRequest.of(pageNum,pageSize,sort(order))));
    }

    private Sort sort(String order){
        String col = "discountPrice";
        return Sort.by(order(order,col));
    }

    private Sort.Order order(String order,String col){
        return  "desc".equals(order) ? desc(col) : asc(col);
    }

    private Sort sort2(String order){
        String[] cols = new String[]{"itemTop","discountPrice","createTime"};
        List<Sort.Order> list = new ArrayList<>();

        for (String col : cols){
            list.add( order(order,col) );
        }

        return Sort.by(list);
    }

    @GetMapping("/queryItemsByLabel")
    public Result queryItemsByLabel(@RequestParam("label") String label,
                                    @RequestParam(value = "pageNum", defaultValue = "0") Integer pageNum,
                                    @RequestParam(value = "pageSize", defaultValue = "6") Integer pageSize,
                                    @RequestParam(value = "order", required = false) String order) {
        return new Result<>(repository.findAllByItemLabelContaining(label, PageRequest.of(pageNum,pageSize,sort(order))));
    }


    @GetMapping("/queryItemSku")
    public Result queryItemSku( @RequestParam(value = "itemId") String itemId,
                               @RequestParam(value = "userId", required = false) String userId) {

        Optional<TbCfStationItem> itemOptional = repository.findById(itemId);

        if( !itemOptional.isPresent() )
            return new Result(ResultCodeEnum.ILLEGAL_ARGUMENT.getCode(),"Item Id is not found");


        List<TbCfCategory> categoryList = categoryRepository.findAllByItemIdOrderByOrderNumAsc(itemId);
        List<TbCfItemSkus> skusList = skuRepository.findAllByItemIdOrderByOrderNumAsc(itemId);

        HashMap<String, Object> map = new HashMap<>();

        TbCfStationItem item = itemOptional.get();
        //商品参数
        List<TbCfItemParam> itemParamList = itemParamRepository.findAllByItemId(itemId);

        map.put("score",item.getTotalScore());
        map.put("isCollection",userId != null && !userId.isEmpty() && collectionRepository.existsByUserIdAndItemId(userId,itemId));
        map.put("optionList",categoryList);
        map.put("itemDetail",skusList);
        map.put("itemInfo", item);
        map.put("itemParam", itemParamList);
        map.put("itemDesc",descRepository.findById(itemId));

        return new Result<>(map);
    }
}
