package com.example.afrishop_v3.quartz;


import com.example.afrishop_v3.enums.DeliveryStatusEnum;
import com.example.afrishop_v3.models.TbCfOrder;
import com.example.afrishop_v3.models.TbCfPlatformOrder;
import com.example.afrishop_v3.models.Time;
import com.example.afrishop_v3.repository.TbCfOrderRepository;
import com.example.afrishop_v3.repository.TbCfPlatformOrderRepository;
import com.example.afrishop_v3.util.HttpClientUtil;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @Author : wudepeng
 * @Date: 2020/05/25
 * @Description:物流任务
 */
@Component
public class LogisticsTask {

    private static Logger logger = LoggerFactory.getLogger(LogisticsTask.class);
    private final TbCfOrderRepository orderRepository;

    private final TbCfPlatformOrderRepository platformOrderRepository;

    @Value("${logistics.track}")
    private String TRACK_API;

    @Value("${logistics.sign}")
    private String SIGN_API;

    @Value("${logistics.stock}")
    private String STOCK_API;

    @Value("${logistics.user}")
    private String USER_NO;

    @Value("${logistics.token}")
    private String TOKEN;

    @Value("${logistics.add_order}")
    private String ADD_ORDER_API;

    public LogisticsTask(TbCfOrderRepository orderRepository, TbCfPlatformOrderRepository platformOrderRepository) {
        this.orderRepository = orderRepository;
        this.platformOrderRepository = platformOrderRepository;
    }

    //===================================================
    //
    //##定时查询意境通物流订单，然后将数据库订单进行相应的修改
    //##物流状态 20：已到达中国仓 40：已到达非洲仓 50：已签收
    //
    //===================================================

    /**
     * 调用意境通API，查询已到达中国仓的订单，修改数据库订单状态为20
     *
     * @throws Exception
     */
    @Scheduled(cron = "0 0 0/2 * * ? ")
    public void chainaWarehouse() throws Exception {
        //封装请求参数
        Map<String, Object> paramMap = getParamMap();
        paramMap.put("incomeid", 855);
        //发送请求
        afterLogistics(STOCK_API, paramMap, DeliveryStatusEnum.ON_LOAD.getValue());
    }

    /**
     * 调用意境通API，查询已到达非洲的订单，修改数据库订单状态为40
     *
     * @throws Exception
     */
    @Scheduled(cron = "0 0 0/2 * * ? ")
    public void africanWarehouse() throws Exception {
        //封装请求参数
        Map<String, Object> paramMap = getParamMap();
        paramMap.put("incomeid", 1170);
        //发送请求
        afterLogistics(STOCK_API, paramMap, DeliveryStatusEnum.ON_AFRICA.getValue());
    }

    /**
     * 调用意境通API，查询已签收的订单，修改数据库订单状态为50
     *
     * @throws Exception
     */
    @Scheduled(cron = "0 0 0/2 * * ? ")
    public void signed() throws Exception {
        Map<String, Object> paramMap = getParamMap();
        afterLogistics(SIGN_API, paramMap, DeliveryStatusEnum.ARRIVALS.getValue());
    }


    //公共处理方法
    public void afterLogistics(String API, Map paramMap, Integer status) throws Exception {
        String body = HttpClientUtil.sendPostWithBodyParameter(API, paramMap);
        JSONObject jsonObject = JSONObject.fromObject(body);
        System.out.println(jsonObject);
        JSONArray jsonArray = jsonObject.getJSONObject("content").getJSONArray("data");
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject object = jsonArray.getJSONObject(i);
            //快递单号
            String expressNumber = object.getString("sid");
            Optional<TbCfPlatformOrder> platformOrder = platformOrderRepository.findFirstBypExpressNumber(expressNumber);
            if (platformOrder.isPresent()) {
                TbCfPlatformOrder cfPlatformOrder = platformOrder.get();
                Optional<TbCfOrder> order = orderRepository.findById(cfPlatformOrder.getOrderId());
                if (order.isPresent()) {
                    TbCfOrder cfOrder = order.get();
                    cfOrder.setDeliveryFlag(status);
                    orderRepository.save(cfOrder);
                }
            }
        }
    }


    //封装公共的请求参数
    public Map<String, Object> getParamMap() throws ParseException {
        Map<String, Object> paramMap = new HashMap<>();
        Time time = getTime();
        paramMap.put("user_no", USER_NO);
        paramMap.put("token", TOKEN);
        logger.info("开始时间：" + time.getStartTime());
        logger.info("结束时间：" + time.getEndTime());
        paramMap.put("startime", time.getStartTime());
        paramMap.put("endtime", time.getEndTime());
        return paramMap;
    }


    //封装时间
    public Time getTime() throws ParseException {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        Time time = new Time();
        Date date = dateFormat.parse(dateFormat.format(new Date()));
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, -2);
        time.setStartTime(dateFormat.format(calendar.getTime()));
        time.setEndTime(dateFormat.format(new Date()));
        return time;
    }


}
