package com.example.afrishop_v3.controllers;

import com.example.afrishop_v3.base.Result;
import com.example.afrishop_v3.enums.ResultCodeEnum;
import com.example.afrishop_v3.models.ItemLabel;
import com.example.afrishop_v3.models.TbCfStationItem;
import com.example.afrishop_v3.repository.ItemLabelRepository;
import com.example.afrishop_v3.repository.TbCfStationItemRepository;
import com.google.cloud.vision.v1.*;
import com.google.cloud.vision.v1.Feature.Type;
import com.google.protobuf.ByteString;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

//Google image search
@RestController
@RequestMapping("search/image")
public class ImageSearchController extends Controller {
    // Imports the Google Cloud client library
    private final TbCfStationItemRepository repository;
    private final ItemLabelRepository labelRepository;

    public ImageSearchController(TbCfStationItemRepository repository, ItemLabelRepository labelRepository) {



        this.repository = repository;
        this.labelRepository = labelRepository;
    }


    //Get url image details (GOOGLE API)
    @GetMapping
    public Result main(@RequestParam("url") String url) {

        return new Result<>(getItemLabels(url));
    }

    //Search by image url
    @GetMapping("searchUrl")
    public Result searchByUrl(@RequestParam("url") String url, @RequestParam(value = "pageNo",defaultValue = "0",required = false) Integer pageNo, @RequestParam(value = "pageSize",required = false,defaultValue = "12") Integer pageSize) {

        List<ItemLabel> itemLabels = getItemLabels(url);
        List<String> collect = itemLabels.stream().map(ItemLabel::getDescription).collect(Collectors.toList());
        String[] strings = collect.toArray(new String[]{});
        Page<TbCfStationItem> itemsByImageSearch = repository.getItemsByImageSearch(strings, (long) (2), PageRequest.of(pageNo, pageSize));
        Map<String,Object> map = new LinkedHashMap<>();
        map.put("items",itemsByImageSearch.toList());
        map.put("criteria",strings);
        return new Result<>(map);
    }

    //Search by uploaded image
    @PostMapping("/upload")
    public Result handleFileUpload(@RequestParam("file") MultipartFile file, @RequestParam(value = "pageNo",defaultValue = "0",required = false) Integer pageNo, @RequestParam(value = "pageSize",required = false,defaultValue = "12") Integer pageSize){
        if( file != null ){
            try {
                List<ItemLabel> itemLabels = getItemLabels(file.getBytes());
                List<String> collect = itemLabels.stream().map(ItemLabel::getDescription).collect(Collectors.toList());
                String[] strings = collect.toArray(new String[]{});
                Page<TbCfStationItem> itemsByImageSearch = repository.getItemsByImageSearch(strings, (long) (strings.length / 2), PageRequest.of(pageNo, pageSize));
                return new Result<>(itemsByImageSearch.toList());
            } catch (IOException e) {
                System.out.println(e.getMessage());
            }
        }
        return new Result<>(new ArrayList<>());
    }

    //Index item image information
    @GetMapping("indexing")
    public Result itemIndexing(@RequestParam("itemId") String itemId) {

       try {
           Optional<TbCfStationItem> byId = repository.findById(itemId);


           if( byId.isPresent() ){
               TbCfStationItem stationItem = byId.get();


               if( labelRepository.existsByItemItemId(itemId) )
                   return new Result<>(stationItem.getLabelList());

               String img = stationItem.catchSingleImage();
               if( img != null) {
                   System.out.println("Searching");
                   List<ItemLabel> itemLabels = getItemLabels(img);
                   itemLabels.forEach(v->v.setItem(stationItem));
                   stationItem.setLabelList(itemLabels);
                   repository.save(stationItem);
                   return new Result<>(itemLabels);
               }
           }
       }catch (Exception e){
           return new Result(e.getMessage());
       }


        return new Result<>(ResultCodeEnum.SERVICE_ERROR.getCode());
    }


    private List<ItemLabel> getItemLabels(String url){

        RestTemplate restTemplate = new RestTemplate();
        byte[] imageBytes = restTemplate.getForObject(url, byte[].class);

        return this.getItemLabels(imageBytes);
    }

    private List<ItemLabel> getItemLabels(byte[] imageBytes) {
        //authExplicit();
        try (ImageAnnotatorClient vision = ImageAnnotatorClient.create()) {


            ByteString imgBytes = ByteString.copyFrom(imageBytes);

            // Builds the image annotation request
            List<AnnotateImageRequest> requests = new ArrayList<>();
            Image img = Image.newBuilder().setContent(imgBytes).build();
            Feature feat = Feature.newBuilder().setType(Type.LABEL_DETECTION).build();
            AnnotateImageRequest request =
                    AnnotateImageRequest.newBuilder().addFeatures(feat).setImage(img).build();
            requests.add(request);

            // Performs label detection on the image file
            BatchAnnotateImagesResponse response = vision.batchAnnotateImages(requests);
            List<AnnotateImageResponse> responses = response.getResponsesList();
            List<ItemLabel> list = new ArrayList<>();


            for (AnnotateImageResponse res : responses) {
                if (res.hasError()) {
                    System.out.format("Error: %s%n", res.getError().getMessage());
                    return new ArrayList<>();
                }

                for (EntityAnnotation ann : res.getLabelAnnotationsList()) {

                    list.add(new ItemLabel(ann.getScore(), ann.getTopicality(), ann.getDescription(), ann.getMid()));
                }

                return list;
            }
        } catch (IOException e) {
            return new ArrayList<>();
        }

        return new ArrayList<>();
    }
}
