package com.example.afrishop_v3.bis.impl;

import com.example.afrishop_v3.bis.IItemSpider;
import com.example.afrishop_v3.enums.PlatformEnum;
import com.example.afrishop_v3.vo.*;
import com.example.afrishop_v3.util.*;


import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

/**
 * 优衣库数据爬虫
 * TODO 读取不到链接
 * @author 爱酱油不爱醋
 */
@Component("uniqloSpider")
public class UniqloSpider implements IItemSpider {
    private static Logger logger = LoggerFactory.getLogger(UniqloSpider.class);

    /**
     * Uniqlo 商品详情链接
     */
    private static final String UNIQLO_URL = "https://www.uniqlo.cn/data/products/spu/";

    /**
     * Uniqlo 数据爬虫
     * @param targetUrl 接收的商品详情路径
     * @return 格式化与翻译后的 Json 数据
     */
    @Override
    public JSONObject captureItem(String targetUrl) throws URISyntaxException, IOException, ExecutionException, InterruptedException, TimeoutException {
        // 截取链接中的商品 id
        String[] split = targetUrl.split("pid=");
        String pId = split[1];
        targetUrl = UNIQLO_URL + "zh_CN/" + pId + ".json";
        String content = HttpClientUtil.getContentByUrl(targetUrl, PlatformEnum.UNIQLO.getValue());
        // 获取折扣价格的接口
        String priceUrl = "https://d.uniqlo.cn/p/product/i/product/spu/pc/query/" + pId + "/zh_CN";
        String priceContent = HttpClientUtil.getContentByUrl(priceUrl, PlatformEnum.UNIQLO.getValue());
        JSONObject resultJson = JSONObject.fromObject(content);
        JSONObject priceJson = JSONObject.fromObject(priceContent);
        ProductResponse productResponse = formatProductResponse(resultJson, priceJson, pId);
        resultJson = JSONObject.fromObject(productResponse);
        // 翻译
        TranslateHelper.translateProductResponse(resultJson);
        return resultJson;
    }

    /**
     * 返回格式化数据
     * @param dataMap 调用优衣库网页接口接收的主要商品数据
     * @param pId 商品链接截取的商品id
     * @return
     */
    private ProductResponse formatProductResponse(JSONObject dataMap, JSONObject priceJson, String pId) {
        // 声明封装类
        ProductResponse productResponse = new ProductResponse();
        // 属性：Zara 的商品属性有颜色、尺码
        Map<String, Set<ProductProp>> productPropSet = new HashMap<>(16);
        // 原始价
        List<OriginalPrice> originalPriceList = new ArrayList<>();
        // 促销价格
        List<ProductPromotion> promotionList = new ArrayList<>();
        // 库存
        DynStock dynStock = new DynStock();
        // 其实数据没有包含确切的库存数，这里默认给足量的库存
        dynStock.setSellableQuantity(9999);
        // 商品基本信息
        ItemInfo itemInfo = new ItemInfo();
        // 取 summary 节点对象
        JSONObject summaryObj = dataMap.getJSONObject("summary");

        //////////////////////////////////// 获取商品基本信息 ////////////////////////////////////////////
        itemInfo.setShopName(PlatformEnum.UNIQLO.getLabel());
        itemInfo.setShopUrl("https://www.uniqlo.cn/product-detail.html");
        itemInfo.setItemId(pId);
        itemInfo.setTitle(summaryObj.getString("fullName"));
        //////////////////////////////////// 获取商品基本信息End(图片下取) ////////////////////////////////////////////

        // 取 rows 节点数组
        JSONArray rowsArr = dataMap.getJSONArray("rows");
        for (int i = 0; i < rowsArr.size(); i++) {
            JSONObject rowsObj = rowsArr.getJSONObject(i);

            //////////////////////////////////// 获取商品颜色属性 ////////////////////////////////////////////
            Set<ProductProp> propSetColor = new HashSet<>(16);
            ProductProp productPropColor = new ProductProp();
            // 颜色id
            String colorNo = rowsObj.getString("colorNo");
            productPropColor.setPropId(colorNo);
            // 颜色名
            String colorName = rowsObj.getString("style");
            productPropColor.setPropName(colorName);
            // 颜色图片
            String imageUrl = "https://www.uniqlo.cn/hmall/test/" + pId + "/sku/561/" + colorNo + ".jpg";
            productPropColor.setImage(imageUrl);
            if (i == 0) {
                itemInfo.setPic(imageUrl);
            }
            propSetColor.add(productPropColor);
            if (productPropSet.get("颜色") == null) {
                productPropSet.put("颜色", propSetColor);
            } else {
                Set<ProductProp> oldPropSet = productPropSet.get("颜色");
                propSetColor.addAll(oldPropSet);
                productPropSet.put("颜色", propSetColor);
            }
            //////////////////////////////////// 获取商品颜色属性 END ////////////////////////////////////////////

            ///////////////////////// 获取商品尺码属性 ////////////////////
            Set<ProductProp> sizePropSetSize = new HashSet<>();
            ProductProp productPropSize = new ProductProp();
            String size = rowsObj.getString("sizeText");
            productPropSize.setPropName(size);
            String sizeNo = rowsObj.getString("skuId");
            productPropSize.setPropId(sizeNo);
            sizePropSetSize.add(productPropSize);
            if (productPropSet.get("尺码") == null) {
                productPropSet.put("尺码", sizePropSetSize);
            } else {
                Set<ProductProp> oldPropSet = productPropSet.get("尺码");
                sizePropSetSize.addAll(oldPropSet);
                productPropSet.put("尺码", sizePropSetSize);
            }
            ///////////////////////// 获取商品尺码属性 END////////////////////

            String skuStr = ";" + colorNo + ";" + sizeNo + ";";

            //////////////////////////////////// 获取库存 ////////////////////////////////////////////
            // 设置：商品包含库存信息
            productResponse.setStockFlag(true);
            List<ProductSkuStock> productSkuStockList = dynStock.getProductSkuStockList();
            if (productSkuStockList == null) {
                productSkuStockList = new ArrayList<>();
            }
            ProductSkuStock productSkuStock = new ProductSkuStock();
            // 设置：可用库存值，Uniqlo 未有可用的库存数据
            productSkuStock.setSellableQuantity(999);
            // 设置：库存对应的id
            productSkuStock.setSkuStr(skuStr);
            productSkuStockList.add(productSkuStock);
            dynStock.setProductSkuStockList(productSkuStockList);
            //////////////////////////////////// 获取库存 END/////////////////////////////////////////

            //////////////////////////////////// 获取原始价 //////////////////////////////////
            OriginalPrice originalPrice = new OriginalPrice();
            // 获取商品的原始价
            String fullPrice = summaryObj.getString("originPrice");
            // TODO 转换汇率，目前商品单位是人民币
            fullPrice = SpiderUtil.exchangeRate(fullPrice);
            originalPrice.setPrice(fullPrice);
            originalPrice.setSkuStr(skuStr);
            originalPriceList.add(originalPrice);
            //////////////////////////////////// 获取原始价 END//////////////////////////////////


            //////////////////////////////////// 获取促销价 //////////////////////////////////
            // 取 summary 节点对象
            JSONObject summaryPriceObj = priceJson.getJSONArray("resp").getJSONObject(0).getJSONObject("summary");

            ProductPromotion productPromotion = new ProductPromotion();
            // 获取商品的促销价
            String promotionPrice = summaryPriceObj.getString("maxPrice");
            // TODO 转换汇率，目前商品单位是人民币
            promotionPrice = SpiderUtil.exchangeRate(promotionPrice);
            productPromotion.setSkuStr(skuStr);
            productPromotion.setPrice(promotionPrice);
            productResponse.setPrice(promotionPrice);
            productResponse.setSalePrice(promotionPrice + "-" + fullPrice);
            promotionList.add(productPromotion);
            //////////////////////////////////// 获取促销价 END//////////////////////////////////

        }

        // 按照一下顺序进行 json 数据的填充
        productResponse.setPropFlag(true);
        productResponse.setPromotionFlag(true);
        productResponse.setProductPropSet(productPropSet);
        productResponse.setPlatform(PlatformEnum.UNIQLO.getValue());
        productResponse.setPromotionList(promotionList);
        productResponse.setOriginalPriceList(originalPriceList);
        productResponse.setItemInfo(itemInfo);
        productResponse.setDynStock(dynStock);
        return productResponse;
    }

}
